<?php

namespace App\Http\Controllers;

use App\Models\Accessoire;
use App\Models\Accessoiremarque;
use App\Models\Categorieaccessoire;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AccessoireController extends Controller
{
    public function index()
    {
        $accessoires = Accessoire::with(['accessoiremarque', 'categorieaccessoire'])
            ->orderBy('ordre', 'desc')
            ->paginate(10);
        return view('admin.accessoires.index', compact('accessoires'));
    }

    public function create()
    {
        $accessoiremarques = Accessoiremarque::where('pub', 1)->orderBy('titre')->get();
        $categorieaccessoires = Categorieaccessoire::where('pub', 1)->orderBy('titre')->get();
        return view('admin.accessoires.form', compact('accessoiremarques', 'categorieaccessoires'));
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'description' => 'nullable|string',
                'content' => 'nullable|string',
                'ref' => 'nullable|string|max:255',
                'prix' => 'required|numeric|min:0',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'pub' => 'boolean',
                'accessoiremarque_id' => 'required|exists:accessoiremarques,id',
                'categorieaccessoire_id' => 'required|exists:categorieaccessoires,id'
            ]);

            $data = $request->all();
            
            if ($request->hasFile('photo')) {
                try {
                    $data['photo'] = $request->file('photo')->store('photos', 'public');
                } catch (\Exception $e) {
                    return redirect()->back()->withInput()->with('error', 'Erreur lors du téléchargement de l\'image: ' . $e->getMessage());
                }
            }

            $accessoire = Accessoire::create([
                'titre' => $request->titre,
                'description' => $request->description,
                'content' => $request->content,
                'ref' => $request->ref,
                'prix' => $request->prix,
                'photo' => $data['photo'] ?? null,
                'pub' => $request->pub ?? 0,
                'accessoiremarque_id' => $request->accessoiremarque_id,
                'categorieaccessoire_id' => $request->categorieaccessoire_id,
                'ordre' => 0
            ]);

            $accessoire->ordre = $accessoire->id;
            $accessoire->save();

            return redirect()->route('admin.accessoires.index')->with('success', 'Accessoire créé avec succès');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Une erreur est survenue lors de la création: ' . $e->getMessage());
        }
    }

    public function edit(Accessoire $accessoire)
    {
        $accessoiremarques = Accessoiremarque::where('pub', 1)->orderBy('titre')->get();
        $categorieaccessoires = Categorieaccessoire::where('pub', 1)->orderBy('titre')->get();
        return view('admin.accessoires.form', compact('accessoire', 'accessoiremarques', 'categorieaccessoires'));
    }

    public function update(Request $request, Accessoire $accessoire)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'description' => 'nullable|string',
                'content' => 'nullable|string',
                'ref' => 'nullable|string|max:255',
                'prix' => 'required|numeric|min:0',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'pub' => 'boolean',
                'accessoiremarque_id' => 'required|exists:accessoiremarques,id',
                'categorieaccessoire_id' => 'required|exists:categorieaccessoires,id',
                'ordre' => 'integer'
            ]);

            $data = $request->all();
            
            if ($request->hasFile('photo')) {
                try {
                    if ($accessoire->photo) {
                        Storage::disk('public')->delete($accessoire->photo);
                    }
                    $data['photo'] = $request->file('photo')->store('photos', 'public');
                } catch (\Exception $e) {
                    return redirect()->back()->withInput()->with('error', 'Erreur lors du téléchargement de l\'image: ' . $e->getMessage());
                }
            }

            $accessoire->update($data);

            return redirect()->route('admin.accessoires.index')->with('success', 'Accessoire modifié avec succès');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Une erreur est survenue lors de la modification: ' . $e->getMessage());
        }
    }

    public function destroy(Accessoire $accessoire)
    {
        try {
            if ($accessoire->photo) {
                Storage::disk('public')->delete($accessoire->photo);
            }
            
            $accessoire->delete();

            return redirect()->route('admin.accessoires.index')->with('success', 'Accessoire supprimé avec succès');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la suppression: ' . $e->getMessage());
        }
    }

    public function moveUp(Accessoire $accessoire)
    {
        $upperAccessoire = Accessoire::where('ordre', '>', $accessoire->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperAccessoire) {
            $tempOrdre = $accessoire->ordre;
            $accessoire->ordre = $upperAccessoire->ordre;
            $upperAccessoire->ordre = $tempOrdre;

            $accessoire->save();
            $upperAccessoire->save();
        }

        return redirect()->route('admin.accessoires.index')->with('success', 'Accessoire déplacé vers le haut');
    }

    public function moveDown(Accessoire $accessoire)
    {
        $lowerAccessoire = Accessoire::where('ordre', '<', $accessoire->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerAccessoire) {
            $tempOrdre = $accessoire->ordre;
            $accessoire->ordre = $lowerAccessoire->ordre;
            $lowerAccessoire->ordre = $tempOrdre;

            $accessoire->save();
            $lowerAccessoire->save();
        }

        return redirect()->route('admin.accessoires.index')->with('success', 'Accessoire déplacé vers le bas');
    }

    public function togglePublication($id)
    {
        $accessoire = Accessoire::findOrFail($id);
        $accessoire->pub = !$accessoire->pub;
        $accessoire->save();

        return redirect()->route('admin.accessoires.index');
    }
}
