<?php

namespace App\Http\Controllers;

use App\Models\Accessoiremarque;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AccessoiremarqueController extends Controller
{
    public function index()
    {
        $accessoiremarques = Accessoiremarque::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.accessoiremarques.index', compact('accessoiremarques'));
    }

    public function create()
    {
        return view('admin.accessoiremarques.form');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'pub' => 'boolean'
            ]);

            $data = $request->all();
            
            if ($request->hasFile('photo')) {
                try {
                    $data['photo'] = $request->file('photo')->store('photos', 'public');
                } catch (\Exception $e) {
                    return redirect()->back()->withInput()->with('error', 'Erreur lors du téléchargement de l\'image: ' . $e->getMessage());
                }
            }

            $accessoiremarque = Accessoiremarque::create([
                'titre' => $request->titre,
                'photo' => $data['photo'] ?? null,
                'pub' => $request->pub ?? 0,
                'ordre' => 0
            ]);

            $accessoiremarque->ordre = $accessoiremarque->id;
            $accessoiremarque->save();

            return redirect()->route('admin.accessoiremarques.index')->with('success', 'Marque accessoire créée avec succès');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Une erreur est survenue lors de la création: ' . $e->getMessage());
        }
    }

    public function edit(Accessoiremarque $accessoiremarque)
    {
        return view('admin.accessoiremarques.form', compact('accessoiremarque'));
    }

    public function update(Request $request, Accessoiremarque $accessoiremarque)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'pub' => 'boolean',
            'ordre' => 'integer'
        ]);

        $data = $request->all();
        
        if ($request->hasFile('photo')) {
            if ($accessoiremarque->photo) {
                Storage::disk('public')->delete($accessoiremarque->photo);
            }
            $data['photo'] = $request->file('photo')->store('photos', 'public');
        }

        $accessoiremarque->update($data);

        return redirect()->route('admin.accessoiremarques.index')->with('success', 'Marque accessoire modifiée avec succès');
    }

    public function destroy(Accessoiremarque $accessoiremarque)
    {
        if ($accessoiremarque->photo) {
            Storage::disk('public')->delete($accessoiremarque->photo);
        }
        
        $accessoiremarque->delete();

        return redirect()->route('admin.accessoiremarques.index')->with('success', 'Marque accessoire supprimée avec succès');
    }

    public function moveUp(Accessoiremarque $accessoiremarque)
    {
        $upperAccessoiremarque = Accessoiremarque::where('ordre', '>', $accessoiremarque->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperAccessoiremarque) {
            $tempOrdre = $accessoiremarque->ordre;
            $accessoiremarque->ordre = $upperAccessoiremarque->ordre;
            $upperAccessoiremarque->ordre = $tempOrdre;

            $accessoiremarque->save();
            $upperAccessoiremarque->save();
        }

        return redirect()->route('admin.accessoiremarques.index')->with('success', 'Marque accessoire déplacée vers le haut');
    }

    public function moveDown(Accessoiremarque $accessoiremarque)
    {
        $lowerAccessoiremarque = Accessoiremarque::where('ordre', '<', $accessoiremarque->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerAccessoiremarque) {
            $tempOrdre = $accessoiremarque->ordre;
            $accessoiremarque->ordre = $lowerAccessoiremarque->ordre;
            $lowerAccessoiremarque->ordre = $tempOrdre;

            $accessoiremarque->save();
            $lowerAccessoiremarque->save();
        }

        return redirect()->route('admin.accessoiremarques.index')->with('success', 'Marque accessoire déplacée vers le bas');
    }

    public function togglePublication($id)
    {
        $accessoiremarque = Accessoiremarque::findOrFail($id);
        $accessoiremarque->pub = !$accessoiremarque->pub;
        $accessoiremarque->save();

        return redirect()->route('admin.accessoiremarques.index');
    }
}
