<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use App\Models\Categorieblog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BlogController extends Controller
{
    public function index(Request $request)
    {
        $query = Blog::with('categorieblog');
        
        if ($request->has('categorie') && $request->categorie) {
            $query->where('categorieblog_id', $request->categorie);
        }
        
        $blogs = $query->orderBy('ordre', 'desc')->paginate(10);
        $categorieblogs = \App\Models\Categorieblog::orderBy('titre')->get();
        
        return view('admin.blogs.index', compact('blogs', 'categorieblogs'));
    }

    public function create()
    {
        $categorieblogs = Categorieblog::where('pub', 1)->orderBy('titre')->get();
        return view('admin.blogs.form', compact('categorieblogs'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'description' => 'nullable|string',
            'contenu' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'categorieblog_id' => 'required|exists:categorieblogs,id',
            'pub' => 'boolean',
            'ordre' => 'nullable|integer',
            'slug' => 'nullable|string|max:255|unique:blogs,slug',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $data = $request->all();
        
        // Générer le slug automatiquement si non fourni
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['titre']);
        }
        
        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('photos', 'public');
        }

        if (!isset($data['ordre']) || empty($data['ordre'])) {
            $data['ordre'] = Blog::max('ordre') + 1;
        }

        Blog::create($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog créé avec succès');
    }

    public function edit(Blog $blog)
    {
        $categorieblogs = Categorieblog::where('pub', 1)->orderBy('titre')->get();
        return view('admin.blogs.form', compact('blog', 'categorieblogs'));
    }

    public function update(Request $request, Blog $blog)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'description' => 'nullable|string',
            'contenu' => 'nullable|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'categorieblog_id' => 'required|exists:categorieblogs,id',
            'pub' => 'boolean',
            'ordre' => 'nullable|integer',
            'slug' => 'nullable|string|max:255|unique:blogs,slug,' . $blog->id,
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $data = $request->all();
        
        // Générer le slug automatiquement si non fourni
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['titre']);
        }
        
        if ($request->hasFile('photo')) {
            if ($blog->photo) {
                Storage::disk('public')->delete($blog->photo);
            }
            $data['photo'] = $request->file('photo')->store('photos', 'public');
        }

        $blog->update($data);

        return redirect()->route('admin.blogs.index')->with('success', 'Blog modifié avec succès');
    }

    public function destroy(Blog $blog)
    {
        if ($blog->photo) {
            Storage::disk('public')->delete($blog->photo);
        }
        
        $blog->delete();

        return redirect()->route('admin.blogs.index')->with('success', 'Blog supprimé avec succès');
    }

    public function moveUp(Blog $blog)
    {
        $previousBlog = Blog::where('ordre', '<', $blog->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($previousBlog) {
            $tempOrdre = $blog->ordre;
            $blog->ordre = $previousBlog->ordre;
            $previousBlog->ordre = $tempOrdre;

            $blog->save();
            $previousBlog->save();
        }

        return redirect()->route('admin.blogs.index')->with('success', 'Ordre mis à jour');
    }

    public function moveDown(Blog $blog)
    {
        $nextBlog = Blog::where('ordre', '>', $blog->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($nextBlog) {
            $tempOrdre = $blog->ordre;
            $blog->ordre = $nextBlog->ordre;
            $nextBlog->ordre = $tempOrdre;

            $blog->save();
            $nextBlog->save();
        }

        return redirect()->route('admin.blogs.index')->with('success', 'Ordre mis à jour');
    }

    public function togglePublication($id)
    {
        $blog = Blog::findOrFail($id);
        $blog->pub = !$blog->pub;
        $blog->save();

        $message = $blog->pub ? 'Blog publié avec succès' : 'Blog dépublié avec succès';
        return redirect()->route('admin.blogs.index')->with('success', $message);
    }
}
