<?php

namespace App\Http\Controllers;

use App\Models\Bordure;
use Illuminate\Http\Request;

class BordureController extends Controller
{
    public function index()
    {
        $bordures = Bordure::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.bordures.index', compact('bordures'));
    }

    public function create()
    {
        return view('admin.bordures.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'prix' => 'required|numeric',
            'photo' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'required|in:0,1'
        ]);

        $photoPath = $request->file('photo')->store('photos', 'public');

        $bordure = Bordure::create([
            'titre' => $request->titre,
            'prix' => $request->prix,
            'photo' => $photoPath,
            'pub' => $request->pub,
            'ordre' => 0
        ]);

        $bordure->ordre = $bordure->id;
        $bordure->save();

        return redirect()->route('admin.bordures.index')->with('success', 'Bordure créée avec succès');
    }

    public function edit(Bordure $bordure)
    {
        return view('admin.bordures.form', compact('bordure'));
    }

    public function update(Request $request, Bordure $bordure)
    {
        $request->validate([
            'titre' => 'required',
            'prix' => 'required|numeric',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'required|in:0,1'
        ]);

        $photoPath = $bordure->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $bordure->update([
            'titre' => $request->titre,
            'prix' => $request->prix,
            'photo' => $photoPath,
            'pub' => $request->pub
        ]);

        return redirect()->route('admin.bordures.edit', $bordure)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Bordure $bordure)
    {
        $bordure->delete();
        return redirect()->route('admin.bordures.index')->with('success', 'Bordure supprimée avec succès');
    }

    public function moveUp(Bordure $bordure)
    {
        $upperBordure = Bordure::where('ordre', '>', $bordure->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperBordure) {
            $tempOrdre = $bordure->ordre;
            $bordure->ordre = $upperBordure->ordre;
            $upperBordure->ordre = $tempOrdre;

            $bordure->save();
            $upperBordure->save();
        }

        return redirect()->route('admin.bordures.index')->with('success', 'Bordure déplacée vers le haut');
    }

    public function moveDown(Bordure $bordure)
    {
        $lowerBordure = Bordure::where('ordre', '<', $bordure->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerBordure) {
            $tempOrdre = $bordure->ordre;
            $bordure->ordre = $lowerBordure->ordre;
            $lowerBordure->ordre = $tempOrdre;

            $bordure->save();
            $lowerBordure->save();
        }

        return redirect()->route('admin.bordures.index')->with('success', 'Bordure déplacée vers le bas');
    }

    public function togglePublication($id)
    {
        $bordure = Bordure::findOrFail($id);
        $bordure->pub = !$bordure->pub;
        $bordure->save();

        return redirect()->route('admin.bordures.index');
    }

    /**
     * Récupérer toutes les bordures pour AJAX
     */
    public function getAllBordures()
    {
        try {
            $bordures = Bordure::where('pub', 1)
                              ->orderBy('ordre')
                              ->get(['id', 'titre', 'prix', 'photo']);
            
            return response()->json($bordures);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erreur lors du chargement des bordures'], 500);
        }
    }
}
