<?php

namespace App\Http\Controllers;

use App\Models\Categorieaccessoire;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CategorieaccessoireController extends Controller
{
    public function index()
    {
        try {
            $categorieaccessoires = Categorieaccessoire::orderBy('ordre', 'desc')->paginate(10);
            return view('admin.categorieaccessoires.index', compact('categorieaccessoires'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors du chargement des catégories: ' . $e->getMessage());
        }
    }

    public function create()
    {
        return view('admin.categorieaccessoires.form');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'pub' => 'boolean',
                'ordre' => 'integer'
            ]);

            $data = $request->all();
            
            if ($request->hasFile('photo')) {
                try {
                    $data['photo'] = $request->file('photo')->store('photos', 'public');
                } catch (\Exception $e) {
                    return redirect()->back()->withInput()->with('error', 'Erreur lors du téléchargement de l\'image: ' . $e->getMessage());
                }
            }

            $categorieaccessoire = Categorieaccessoire::create([
                'titre' => $request->titre,
                'photo' => $data['photo'] ?? null,
                'pub' => $request->pub ?? 0,
                'ordre' => 0
            ]);

            $categorieaccessoire->ordre = $categorieaccessoire->id;
            $categorieaccessoire->save();

            return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire créée avec succès');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Une erreur est survenue lors de la création: ' . $e->getMessage());
        }
    }

    public function edit(Categorieaccessoire $categorieaccessoire)
    {
        return view('admin.categorieaccessoires.form', compact('categorieaccessoire'));
    }

    public function update(Request $request, Categorieaccessoire $categorieaccessoire)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'pub' => 'boolean',
                'ordre' => 'integer'
            ]);

            $data = $request->all();
            
            if ($request->hasFile('photo')) {
                try {
                    // Supprimer l'ancienne photo si elle existe
                    if ($categorieaccessoire->photo && Storage::disk('public')->exists($categorieaccessoire->photo)) {
                        Storage::disk('public')->delete($categorieaccessoire->photo);
                    }
                    $data['photo'] = $request->file('photo')->store('photos', 'public');
                } catch (\Exception $e) {
                    return redirect()->back()->withInput()->with('error', 'Erreur lors du téléchargement de l\'image: ' . $e->getMessage());
                }
            }

            $categorieaccessoire->update($data);

            return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire modifiée avec succès');
        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()->withErrors($e->validator)->withInput();
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Une erreur est survenue lors de la modification: ' . $e->getMessage());
        }
    }

    public function destroy(Categorieaccessoire $categorieaccessoire)
    {
        try {
            // Supprimer l'image associée si elle existe
            if ($categorieaccessoire->photo && Storage::disk('public')->exists($categorieaccessoire->photo)) {
                try {
                    Storage::disk('public')->delete($categorieaccessoire->photo);
                } catch (\Exception $e) {
                    \Log::warning('Erreur lors de la suppression de l\'image: ' . $e->getMessage());
                }
            }
            
            $categorieaccessoire->delete();

            return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire supprimée avec succès');
        } catch (\Exception $e) {
            return redirect()->route('admin.categorieaccessoires.index')->with('error', 'Une erreur est survenue lors de la suppression: ' . $e->getMessage());
        }
    }

    public function moveUp(Categorieaccessoire $categorieaccessoire)
    {
        try {
            $upperCategorieaccessoire = Categorieaccessoire::where('ordre', '>', $categorieaccessoire->ordre)
                ->orderBy('ordre', 'asc')
                ->first();

            if ($upperCategorieaccessoire) {
                $tempOrdre = $categorieaccessoire->ordre;
                $categorieaccessoire->ordre = $upperCategorieaccessoire->ordre;
                $upperCategorieaccessoire->ordre = $tempOrdre;

                $categorieaccessoire->save();
                $upperCategorieaccessoire->save();
                
                return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire déplacée vers le haut');
            }

            return redirect()->route('admin.categorieaccessoires.index')->with('info', 'La catégorie est déjà en première position');
        } catch (\Exception $e) {
            return redirect()->route('admin.categorieaccessoires.index')->with('error', 'Erreur lors du déplacement: ' . $e->getMessage());
        }
    }

    public function moveDown(Categorieaccessoire $categorieaccessoire)
    {
        try {
            $lowerCategorieaccessoire = Categorieaccessoire::where('ordre', '<', $categorieaccessoire->ordre)
                ->orderBy('ordre', 'desc')
                ->first();

            if ($lowerCategorieaccessoire) {
                $tempOrdre = $categorieaccessoire->ordre;
                $categorieaccessoire->ordre = $lowerCategorieaccessoire->ordre;
                $lowerCategorieaccessoire->ordre = $tempOrdre;

                $categorieaccessoire->save();
                $lowerCategorieaccessoire->save();
                
                return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire déplacée vers le bas');
            }

            return redirect()->route('admin.categorieaccessoires.index')->with('info', 'La catégorie est déjà en dernière position');
        } catch (\Exception $e) {
            return redirect()->route('admin.categorieaccessoires.index')->with('error', 'Erreur lors du déplacement: ' . $e->getMessage());
        }
    }

    public function togglePublication($id)
    {
        try {
            $categorieaccessoire = Categorieaccessoire::findOrFail($id);
            $categorieaccessoire->pub = !$categorieaccessoire->pub;
            $categorieaccessoire->save();

            $status = $categorieaccessoire->pub ? 'publiée' : 'dépubliée';
            return redirect()->route('admin.categorieaccessoires.index')->with('success', 'Catégorie accessoire ' . $status . ' avec succès');
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return redirect()->route('admin.categorieaccessoires.index')->with('error', 'Catégorie accessoire introuvable');
        } catch (\Exception $e) {
            return redirect()->route('admin.categorieaccessoires.index')->with('error', 'Erreur lors du changement de statut: ' . $e->getMessage());
        }
    }
}
