<?php

namespace App\Http\Controllers;

use App\Models\Categoriematiere;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategorimatiereController extends Controller
{
    public function index()
    {
        $categoriematieres = Categoriematiere::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.categoriematieres.index', compact('categoriematieres'));
    }

    public function create()
    {
        return view('admin.categoriematieres.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        // Générer le slug automatiquement si le champ slug est vide
        $slug = $request->slug;
        if (empty($slug)) {
            $slug = Str::slug($request->titre);
            
            // Vérifier l'unicité du slug
            $originalSlug = $slug;
            $counter = 1;
            while (Categoriematiere::where('slug', $slug)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        $categoriematiere = Categoriematiere::create([
            'titre' => $request->titre,
            'description' => $request->description,
            'slug' => $slug,
            'pub' => $request->pub,
            'photo' => $photoPath,
            'ordre' => 0,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        $categoriematiere->ordre = $categoriematiere->id;
        $categoriematiere->save();

        return redirect()->route('admin.categoriematieres.index')->with('success', 'Catégorie matière créée avec succès');
    }

    public function edit(Categoriematiere $categoriematiere)
    {
        return view('admin.categoriematieres.form', compact('categoriematiere'));
    }

    public function update(Request $request, Categoriematiere $categoriematiere)
    {
        $request->validate([
            'titre' => 'required',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $photoPath = $categoriematiere->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        // Générer le slug automatiquement si le champ slug est vide
        $slug = $request->slug;
        if (empty($slug)) {
            $slug = Str::slug($request->titre);
            
            // Vérifier l'unicité du slug (en excluant la catégorie actuelle)
            $originalSlug = $slug;
            $counter = 1;
            while (Categoriematiere::where('slug', $slug)->where('id', '!=', $categoriematiere->id)->exists()) {
                $slug = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        $categoriematiere->update([
            'titre' => $request->titre,
            'description' => $request->description,
            'slug' => $slug,
            'pub' => $request->pub,
            'photo' => $photoPath,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        return redirect()->route('admin.categoriematieres.edit', $categoriematiere)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Categoriematiere $categoriematiere)
    {
        $categoriematiere->delete();
        return redirect()->route('admin.categoriematieres.index')->with('success', 'Catégorie matière supprimée avec succès');
    }

    public function moveUp(Categoriematiere $categoriematiere)
    {
        $upperItem = Categoriematiere::where('ordre', '>', $categoriematiere->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperItem) {
            $tempOrdre = $categoriematiere->ordre;
            $categoriematiere->ordre = $upperItem->ordre;
            $upperItem->ordre = $tempOrdre;

            $categoriematiere->save();
            $upperItem->save();
        }

        return redirect()->route('admin.categoriematieres.index')->with('success', 'Ordre modifié avec succès');
    }

    public function moveDown(Categoriematiere $categoriematiere)
    {
        $lowerItem = Categoriematiere::where('ordre', '<', $categoriematiere->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerItem) {
            $tempOrdre = $categoriematiere->ordre;
            $categoriematiere->ordre = $lowerItem->ordre;
            $lowerItem->ordre = $tempOrdre;

            $categoriematiere->save();
            $lowerItem->save();
        }

        return redirect()->route('admin.categoriematieres.index')->with('success', 'Ordre modifié avec succès');
    }

    public function togglePublication($id)
    {
        $categoriematiere = Categoriematiere::findOrFail($id);
        $categoriematiere->pub = !$categoriematiere->pub;
        $categoriematiere->save();

        return redirect()->route('admin.categoriematieres.index');
    }
}
