<?php

namespace App\Http\Controllers;

use App\Models\Colori;
use App\Models\Matiere;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ColoriController extends Controller
{
    public function index(Request $request)
    {
        $query = Colori::with('matiere');

        if ($request->has('matiere_id') && $request->matiere_id) {
            $query->where('matiere_id', $request->matiere_id);
        }

        $coloris = $query->orderBy('ordre', 'desc')->paginate(10);
        $selectedMatiere = $request->matiere_id ? \App\Models\Matiere::find($request->matiere_id) : null;

        // Conserver le filtre dans la pagination
        $coloris->appends($request->only('matiere_id'));

        return view('admin.coloris.index', compact('coloris', 'selectedMatiere'));
    }

    public function create(Request $request)
    {
        $matieres = Matiere::orderBy('titre')->get();
        $finitions = \App\Models\Finition::orderBy('titre')->get();
        $epaisseurs = \App\Models\Epaisseur::orderBy('titre')->get();
        $selectedMatiereId = $request->get('matiere_id');
        $selectedMatiere = $selectedMatiereId ? Matiere::find($selectedMatiereId) : null;

        return view('admin.coloris.form', compact('matieres', 'finitions', 'epaisseurs', 'selectedMatiereId', 'selectedMatiere'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'description' => 'required|string',
            'matiere_id' => 'required|exists:matieres,id',
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'pub' => 'boolean',
            'ordre' => 'integer',
            'tarifs' => 'nullable|array',
            'tarifs.*.finition_id' => 'nullable|exists:finitions,id',
            'tarifs.*.epaisseur_id' => 'nullable|exists:epaisseurs,id',
            'tarifs.*.prix' => 'nullable|numeric|min:0'
        ]);

        $data = $request->all();

        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('photos', 'public');
        }

        if (!isset($data['ordre'])) {
            $data['ordre'] = Colori::max('ordre') + 1;
        }

        $colori = Colori::create($data);

        // Gestion des images multiples
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $imagePath = $image->store('photos', 'public');
                $colori->photocoloris()->create([
                    'photo' => $imagePath
                ]);
            }
        }

        // Gestion des tarifs
        if ($request->has('tarifs')) {
            foreach ($request->tarifs as $tarif) {
                if (!empty($tarif['finition_id']) && !empty($tarif['epaisseur_id']) && !empty($tarif['prix'])) {
                    // Vérifier si cette combinaison existe déjà
                    if (\App\Models\Prixcolori::combinaisonExiste($colori->id, $tarif['finition_id'], $tarif['epaisseur_id'])) {
                        return redirect()->back()->withErrors(['tarifs' => 'Cette combinaison finition/épaisseur existe déjà pour ce coloris.'])->withInput();
                    }

                    $colori->prixcoloris()->create([
                        'finition_id' => $tarif['finition_id'],
                        'epaisseur_id' => $tarif['epaisseur_id'],
                        'prix' => $tarif['prix']
                    ]);
                }
            }
        }

        return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id])->with('success', 'Colori créé avec succès');
    }

    public function edit(Colori $colori)
    {
        $colori->load(['photocoloris', 'prixcoloris.finition', 'prixcoloris.epaisseur']);
        $matieres = Matiere::orderBy('titre')->get();
        $finitions = \App\Models\Finition::orderBy('titre')->get();
        $epaisseurs = \App\Models\Epaisseur::orderBy('titre')->get();
        return view('admin.coloris.form', compact('colori', 'matieres', 'finitions', 'epaisseurs'));
    }

    public function update(Request $request, Colori $colori)
    {
        // Vérifier le nombre total de photos (existantes + nouvelles)
        $photosExistantes = $colori->photocoloris()->count();
        $nouvellesPhotos = $request->hasFile('images') ? count($request->file('images')) : 0;
        $totalPhotos = $photosExistantes + $nouvellesPhotos;

        if ($totalPhotos > 10) {
            return redirect()->back()->withErrors(['images' => 'Le nombre total de photos ne peut pas dépasser 10. Vous avez actuellement ' . $photosExistantes . ' photos.'])->withInput();
        }

        $request->validate([
            'matiere_id' => 'required|exists:matieres,id',
            'titre' => 'required|string|max:255',
            'description' => 'required|string',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'pub' => 'boolean',
            'slug' => 'nullable|string|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'images' => 'nullable|array|max:10',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'tarifs' => 'nullable|array',
            'tarifs.*.finition_id' => 'nullable|exists:finitions,id',
            'tarifs.*.epaisseur_id' => 'nullable|exists:epaisseurs,id',
            'tarifs.*.prix' => 'nullable|numeric|min:0'
        ]);

        $data = $request->all();

        if ($request->hasFile('photo')) {
            if ($colori->photo) {
                Storage::disk('public')->delete($colori->photo);
            }
            $data['photo'] = $request->file('photo')->store('photos', 'public');
        }

        $colori->update($data);

        // Gestion des nouvelles images multiples
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $imagePath = $image->store('photos', 'public');
                $colori->photocoloris()->create([
                    'photo' => $imagePath
                ]);
            }
        }

        // Gestion des tarifs
        if ($request->has('tarifs')) {
            // Collecte des IDs des tarifs soumis pour savoir lesquels supprimer
            $tarifsIds = [];

            foreach ($request->tarifs as $tarif) {
                if (!empty($tarif['finition_id']) && !empty($tarif['epaisseur_id']) && !empty($tarif['prix'])) {
                    if (!empty($tarif['prix_id'])) {
                        // Modification d'un tarif existant
                        $prixColori = $colori->prixcoloris()->find($tarif['prix_id']);
                        if ($prixColori) {
                            // Vérifier si cette combinaison existe déjà (exclure le tarif actuel)
                            if (\App\Models\Prixcolori::combinaisonExiste($colori->id, $tarif['finition_id'], $tarif['epaisseur_id'], $prixColori->id)) {
                                return redirect()->back()->withErrors(['tarifs' => 'Cette combinaison finition/épaisseur existe déjà pour ce coloris.'])->withInput();
                            }

                            $prixColori->update([
                                'finition_id' => $tarif['finition_id'],
                                'epaisseur_id' => $tarif['epaisseur_id'],
                                'prix' => $tarif['prix']
                            ]);
                            $tarifsIds[] = $prixColori->id;
                        }
                    } else {
                        // Nouveau tarif - vérifier si cette combinaison existe déjà
                        if (\App\Models\Prixcolori::combinaisonExiste($colori->id, $tarif['finition_id'], $tarif['epaisseur_id'])) {
                            return redirect()->back()->withErrors(['tarifs' => 'Cette combinaison finition/épaisseur existe déjà pour ce coloris.'])->withInput();
                        }

                        $newPrix = $colori->prixcoloris()->create([
                            'finition_id' => $tarif['finition_id'],
                            'epaisseur_id' => $tarif['epaisseur_id'],
                            'prix' => $tarif['prix']
                        ]);
                        $tarifsIds[] = $newPrix->id;
                    }
                }
            }

            // Supprimer les tarifs qui ne sont plus dans la liste
            $colori->prixcoloris()->whereNotIn('id', $tarifsIds)->delete();
        } else {
            // Si aucun tarif n'est soumis, supprimer tous les tarifs existants
            $colori->prixcoloris()->delete();
        }

        return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id])->with('success', 'Colori modifié avec succès');
    }

    public function destroy(Colori $colori)
    {
        try {
            $matiereId = $colori->matiere_id;

            if ($colori->photo) {
                Storage::disk('public')->delete($colori->photo);
            }

            $colori->delete();

            return redirect()->route('admin.coloris.index', ['matiere_id' => $matiereId])->with('success', 'Colori supprimé avec succès');
        } catch (\Exception $e) {
            return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id])->with('error', 'Erreur lors de la suppression du colori');
        }
    }

    public function moveUp(Colori $colori)
    {
        $previousColori = Colori::where('matiere_id', $colori->matiere_id)
                                ->where('ordre', '>', $colori->ordre)
                                ->orderBy('ordre', 'asc')
                                ->first();

        if ($previousColori) {
            $tempOrdre = $colori->ordre;
            $colori->ordre = $previousColori->ordre;
            $previousColori->ordre = $tempOrdre;

            $colori->save();
            $previousColori->save();
        }

        return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id]);
    }

    public function moveDown(Colori $colori)
    {
        $nextColori = Colori::where('matiere_id', $colori->matiere_id)
                            ->where('ordre', '<', $colori->ordre)
                            ->orderBy('ordre', 'desc')
                            ->first();

        if ($nextColori) {
            $tempOrdre = $colori->ordre;
            $colori->ordre = $nextColori->ordre;
            $nextColori->ordre = $tempOrdre;

            $colori->save();
            $nextColori->save();
        }

        return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id]);
    }

    public function togglePublication($id)
    {
        $colori = Colori::findOrFail($id);
        $colori->pub = !$colori->pub;
        $colori->save();

        return redirect()->route('admin.coloris.index', ['matiere_id' => $colori->matiere_id]);
    }

    public function getByMatiere($matiereId)
    {
        try {
            \Log::info("Récupération des coloris pour la matière ID: " . $matiereId);

            $coloris = Colori::where('matiere_id', $matiereId)
                             ->where('pub', 1)
                             ->orderBy('titre')
                             ->select('id', 'titre')
                             ->get();

            \Log::info("Coloris trouvés: " . $coloris->count());
            \Log::info("Données coloris: " . $coloris->toJson());

            return response()->json($coloris);

        } catch (\Exception $e) {
            \Log::error("Erreur lors de la récupération des coloris: " . $e->getMessage());
            return response()->json(['error' => 'Erreur lors de la récupération des coloris'], 500);
        }
    }

    public function getPhoto($coloriId)
    {
        try {
            $colori = Colori::find($coloriId);

            if (!$colori) {
                return response()->json(['error' => 'Coloris non trouvé'], 404);
            }

            // Récupérer le champ photo principal du coloris
            if ($colori->photo) {
                return response()->json(['photo' => $colori->photo]);
            } else {
                // Si pas de photo principale, récupérer la première photo des photocoloris
                $colori->load('photocoloris');
                $photo = $colori->photocoloris->first();
                
                if ($photo) {
                    return response()->json(['photo' => $photo->photo]);
                } else {
                    return response()->json(['photo' => null]);
                }
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erreur lors du chargement de la photo'], 500);
        }
    }
}