<?php

namespace App\Http\Controllers;

use App\Models\Devi;
use App\Models\Client;
use App\Models\Matiere;
use App\Models\Colori;
use App\Models\Finition;
use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Models\Bordure;
use App\Models\Optionsdecoupe;
use Barryvdh\DomPDF\Facade\Pdf;

class DeviController extends Controller
{
    public function index(Request $request)
    {
        $query = Devi::with(['client', 'matiere', 'colori', 'finition']);

        // Search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('id', 'like', "%{$search}%")
                  ->orWhere('devis_pdf', 'like', "%{$search}%")
                  ->orWhereHas('client', function($clientQuery) use ($search) {
                      $clientQuery->where('nom', 'like', "%{$search}%")
                                  ->orWhere('prenom', 'like', "%{$search}%")
                                  ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        // Date range filter
        if ($request->filled('date_range')) {
            $dates = explode(' - ', $request->date_range);
            if (count($dates) == 2) {
                $startDate = \Carbon\Carbon::createFromFormat('d/m/Y', trim($dates[0]))->startOfDay();
                $endDate = \Carbon\Carbon::createFromFormat('d/m/Y', trim($dates[1]))->endOfDay();
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }
        }

        // Client filter
        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }

        // Status filter
        if ($request->filled('statut')) {
            $query->where('statut', $request->statut);
        }

        $devis = $query->orderBy('id', 'desc')->paginate(10);

        // Calculate total amount for current filtered results
        $totalAmount = $query->sum('total_ht');

        // Get all clients for filter dropdown
        $clients = Client::orderBy('nom')->get();

        return view('admin.devis.index', compact('devis', 'totalAmount', 'clients'));
    }

    public function create()
    {
        $clients = Client::all();
        $matieres = Matiere::where('pub', 1)->orderBy('titre')->get();
        $coloris = collect(); // Collection vide, sera remplie via AJAX
        $finitions = Finition::where('pub', 1)->orderBy('titre')->get();
        $bordures = Bordure::where('pub', 1)->orderBy('ordre')->get();
        $optionsdecoupes = Optionsdecoupe::where('pub', 1)->orderBy('ordre')->get();
        $services = \App\Models\Service::where('pub', 1)->orderBy('ordre', 'desc')->get();

        return view('admin.devis.form', compact('clients', 'matieres', 'coloris', 'finitions', 'bordures', 'optionsdecoupes', 'services'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'matiere_id' => 'required|exists:matieres,id',
            'colori_id' => 'required|exists:coloris,id',
            'finition_id' => 'required|exists:finitions,id',
            'bordure_id' => 'required|exists:bordures,id',
            'client-prenom' => 'required|string|max:255',
            'client-nom' => 'required|string|max:255',
            'client-email' => 'required|email|max:255',
            'client-tel' => 'required|string|max:20',
            'client-adresse' => 'required|string',
            'client-pays' => 'required|string|max:255',
            'client-ville' => 'required|string|max:255',
            'client-cp' => 'required|string|max:10',
            'addperc' => 'required|integer|min:0|max:100',
            'tva' => 'nullable|boolean',
            'kt_docs_repeater_plantravail' => 'nullable|array',
            'kt_docs_repeater_plantravail.*.pl-longueur' => 'nullable|numeric',
            'kt_docs_repeater_plantravail.*.pl-largeur' => 'nullable|numeric',
            'kt_docs_repeater_plantravail.*.pl-epaisseur' => 'nullable|exists:prixcoloris,id',
                           'kt_docs_repeater_credence' => 'nullable|array',
                           'kt_docs_repeater_credence.*.credence_longueur' => 'nullable|numeric',
                           'kt_docs_repeater_credence.*.credence_largeur' => 'nullable|numeric',
                           'kt_docs_repeater_credence.*.credence_epaisseur' => 'nullable|exists:prixcoloris,id',
            'optionsdecoupes' => 'nullable|array',
            'optionsdecoupes.*' => 'nullable|integer|min:0',
                           'services' => 'nullable|array',
                           'services.*' => 'nullable|integer|min:0'


        ]);

        try {
            // Créer le client
            $client = Client::create([
                'prenom' => $request->input('client-prenom'),
                'nom' => $request->input('client-nom'),
                'email' => $request->input('client-email'),
                'telephone' => $request->input('client-tel'),
                'adresse' => $request->input('client-adresse'),
                'ville' => $request->input('client-ville'),
                'code_postal' => $request->input('client-cp'),
            ]);

            // Créer le devis
            $devis = Devi::create([
                'client_id' => $client->id,
                'matiere_id' => $request->matiere_id,
                'colori_id' => $request->colori_id,
                'finition_id' => $request->finition_id,
                'bordure_id' => $request->bordure_id,
                'addperc' => $request->addperc,
                'statut' => 0, // Brouillon par défaut
                'devis_pdf' => '', // Vide pour le moment
                'total_ht' => 0, // Sera calculé plus tard
                'tva' => $request->has('tva') ? 10 : 20
            ]);

            // Enregistrer les plans de travail dans devi_prixcolori
            if ($request->has('kt_docs_repeater_plantravail') && is_array($request->kt_docs_repeater_plantravail)) {
                foreach ($request->kt_docs_repeater_plantravail as $plan) {
                    if (isset($plan['pl-longueur']) && isset($plan['pl-largeur']) && isset($plan['pl-epaisseur'])) {
                        // Calculer le montant (longueur * largeur * prix de l'épaisseur)
                        $prixcolori = \App\Models\Prixcolori::find($plan['pl-epaisseur']);
                        $montant = 0;
                        if ($prixcolori) {
                            $surface = ($plan['pl-longueur'] / 1000) * ($plan['pl-largeur'] / 1000); // conversion mm vers m
                            $montant = $surface * $prixcolori->prix;
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($request->addperc / 100));
                        }

                        \App\Models\DeviPrixcolori::create([
                            'devi_id' => $devis->id,
                            'longeur' => $plan['pl-longueur'],
                            'largeur' => $plan['pl-largeur'],
                            'prixcolori_id' => $plan['pl-epaisseur'],
                            'montant' => $montant,
                            'type' => 'pldt'
                        ]);
                    }
                }
            }


            // Enregistrer les plans de travail dans devi_prixcolori
            if ($request->has('kt_docs_repeater_credence') && is_array($request->kt_docs_repeater_credence)) {
                foreach ($request->kt_docs_repeater_credence as $plan) {
                    if (isset($plan['credence_longueur']) && isset($plan['credence_largeur']) && isset($plan['credence_epaisseur'])) {
                        // Calculer le montant (longueur * largeur * prix de l'épaisseur)
                        $prixcolori = \App\Models\Prixcolori::find($plan['credence_epaisseur']);
                        $montant = 0;
                        if ($prixcolori) {
                            $surface = ($plan['credence_longueur'] / 1000) * ($plan['credence_largeur'] / 1000); // conversion mm vers m
                            $montant = $surface * $prixcolori->prix;
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($request->addperc / 100));
                        }

                        \App\Models\DeviPrixcolori::create([
                            'devi_id' => $devis->id,
                            'longeur' => $plan['credence_longueur'],
                            'largeur' => $plan['credence_largeur'],
                            'prixcolori_id' => $plan['credence_epaisseur'],
                            'montant' => $montant,
                            'type' => 'credence'
                        ]);
                    }
                }
            }
                        // Enregistrer les options de decoupe
                        if ($request->has('optionsdecoupes') && is_array($request->optionsdecoupes)) {
                            foreach ($request->optionsdecoupes as $optionsdecoupe_id => $quantite) {
                                if ($quantite > 0) {
                                    // Récupérer le prix de l'option depuis la base de données
                                    $optionsdecoupe = \App\Models\Optionsdecoupe::find($optionsdecoupe_id);
                                    if ($optionsdecoupe) {
                                        $prixu = $optionsdecoupe->prix;
                                        $montant = $quantite * $prixu;
                                        // Appliquer le pourcentage addperc
                                        $montant = $montant * (1 + ($request->addperc / 100));

                                        \App\Models\DeviOptionsdecoupe::create([
                                            'devi_id' => $devis->id,
                                            'optionsdecoupe_id' => $optionsdecoupe_id,
                                            'quantite' => $quantite,
                                            'prixu' => $prixu,
                                            'montant' => $montant,
                                        ]);
                                    }
                                }
                            }
                        }

            // Enregistrer les services
            if ($request->has('services') && is_array($request->services)) {
                foreach ($request->services as $service_id => $value) {
                    if ($value) {
                        // Récupérer le prix du service depuis la base de données
                        $service = \App\Models\Service::find($service_id);
                        if ($service) {
                            \App\Models\DeviService::create([
                                'devi_id' => $devis->id,
                                'service_id' => $service_id,
                                'devis_prix' => $service->prix
                            ]);
                        }
                    }
                }
            }

            // Calculer le montant HT total
            $totalHT = 0;

            // Ajouter les montants des plans de travail et crédences
            $prixcoloris = \App\Models\DeviPrixcolori::where('devi_id', $devis->id)->get();
            foreach ($prixcoloris as $prixcolori) {
                $totalHT += $prixcolori->montant;
            }

            // Ajouter les montants des options de découpe
            $optionsDecoupes = \App\Models\DeviOptionsdecoupe::where('devi_id', $devis->id)->get();
            foreach ($optionsDecoupes as $optionDecoupe) {
                $totalHT += $optionDecoupe->montant;
            }

            // Ajouter les prix des services
            $services = \App\Models\DeviService::where('devi_id', $devis->id)->get();
            foreach ($services as $service) {
                $totalHT += $service->devis_prix;
            }

            // Ajouter le prix de la bordure si elle a un prix
            if ($request->bordure_id && $request->bordure_id !== '0') {
                $bordure = \App\Models\Bordure::find($request->bordure_id);
                if ($bordure && $bordure->prix > 0) {
                    $montantBordure = floatval($bordure->prix);
                    // Appliquer le pourcentage addperc
                    $montantBordure = $montantBordure * (1 + ($request->addperc / 100));
                    $totalHT += $montantBordure;
                }
            }

            // Mettre à jour le devis avec le montant HT calculé
            $devis->update([
                'total_ht' => $totalHT
            ]);

            return redirect()->route('admin.devis.index')->with('success', 'Devis créé avec succès');
        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Erreur lors de la création: ' . $e->getMessage());
        }
    }

    public function show(Devi $devi)
    {
        $devi->load([
            'client', 'matiere', 'colori', 'finition',
            'optionsdecoupes.optionsdecoupe',
            'services.service',
            'prixcoloris.prixcolori',
            'accessoires.accessoire',
            'bordures.bordure'
        ]);

        return view('admin.devis.show', compact('devi'));
    }

    public function edit(Devi $devi)
    {
        // Charger le devis avec toutes ses relations
        $devi->load([
            'client', 'matiere', 'colori', 'finition', 'bordure',
            'prixcoloris.prixcolori',
            'optionsdecoupes.optionsdecoupe',
            'services.service'
        ]);

        $clients = Client::all();
        $matieres = Matiere::where('pub', 1)->orderBy('titre')->get();
        // Charger tous les coloris de la matière sélectionnée
        $coloris = Colori::where('pub', 1)
                         ->where('matiere_id', $devi->matiere_id)
                         ->orderBy('titre')
                         ->get();
        $finitions = Finition::where('pub', 1)->orderBy('titre')->get();
        $bordures = Bordure::where('pub', 1)->orderBy('ordre')->get();
        $optionsdecoupes = Optionsdecoupe::where('pub', 1)->orderBy('ordre')->get();
        $services = \App\Models\Service::where('pub', 1)->orderBy('ordre', 'desc')->get();

        return view('admin.devis.form', compact('devi', 'clients', 'matieres', 'coloris', 'finitions', 'bordures', 'optionsdecoupes', 'services'));
    }

    public function update(Request $request, Devi $devi)
    {
        $request->validate([
            'matiere_id' => 'required|exists:matieres,id',
            'colori_id' => 'required|exists:coloris,id',
            'finition_id' => 'required|exists:finitions,id',
            'bordure_id' => 'required|exists:bordures,id',
            'client-prenom' => 'required|string|max:255',
            'client-nom' => 'required|string|max:255',
            'client-email' => 'required|email|max:255',
            'client-tel' => 'required|string|max:20',
            'client-adresse' => 'required|string',
            'client-pays' => 'required|string|max:255',
            'client-ville' => 'required|string|max:255',
            'client-cp' => 'required|string|max:10',
            'addperc' => 'required|integer|min:0|max:100',
            'tva' => 'nullable|boolean',
            'kt_docs_repeater_plantravail' => 'nullable|array',
            'kt_docs_repeater_plantravail.*.pl-longueur' => 'nullable|numeric',
            'kt_docs_repeater_plantravail.*.pl-largeur' => 'nullable|numeric',
            'kt_docs_repeater_plantravail.*.pl-epaisseur' => 'nullable|exists:prixcoloris,id',
            'kt_docs_repeater_credence' => 'nullable|array',
            'kt_docs_repeater_credence.*.credence_longueur' => 'nullable|numeric',
            'kt_docs_repeater_credence.*.credence_largeur' => 'nullable|numeric',
            'kt_docs_repeater_credence.*.credence_epaisseur' => 'nullable|exists:prixcoloris,id',
            'optionsdecoupes' => 'nullable|array',
            'optionsdecoupes.*' => 'nullable|integer|min:0',
            'services' => 'nullable|array',
            'services.*' => 'nullable|integer|min:0'
        ]);

        try {
            // Mettre à jour le client
            $devi->client->update([
                'prenom' => $request->input('client-prenom'),
                'nom' => $request->input('client-nom'),
                'email' => $request->input('client-email'),
                'telephone' => $request->input('client-tel'),
                'adresse' => $request->input('client-adresse'),
                'ville' => $request->input('client-ville'),
                'code_postal' => $request->input('client-cp'),
            ]);

            // Mettre à jour le devis
            $devi->update([
                'matiere_id' => $request->matiere_id,
                'colori_id' => $request->colori_id,
                'finition_id' => $request->finition_id,
                'bordure_id' => $request->bordure_id,
                'tva' => $request->has('tva') ? 10 : 20,
                'addperc' => $request->addperc
            ]);

            // Supprimer les anciennes données liées
            $devi->prixcoloris()->delete();
            $devi->optionsdecoupes()->delete();
            $devi->services()->delete();

            // Réenregistrer les plans de travail dans devi_prixcolori
            if ($request->has('kt_docs_repeater_plantravail') && is_array($request->kt_docs_repeater_plantravail)) {
                foreach ($request->kt_docs_repeater_plantravail as $plan) {
                    if (isset($plan['pl-longueur']) && isset($plan['pl-largeur']) && isset($plan['pl-epaisseur'])) {
                        $prixcolori = \App\Models\Prixcolori::find($plan['pl-epaisseur']);
                        $montant = 0;
                        if ($prixcolori) {
                            $surface = ($plan['pl-longueur'] / 1000) * ($plan['pl-largeur'] / 1000);
                            $montant = $surface * $prixcolori->prix;
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($request->addperc / 100));
                        }

                        \App\Models\DeviPrixcolori::create([
                            'devi_id' => $devi->id,
                            'longeur' => $plan['pl-longueur'],
                            'largeur' => $plan['pl-largeur'],
                            'prixcolori_id' => $plan['pl-epaisseur'],
                            'montant' => $montant,
                            'type' => 'pldt'
                        ]);
                    }
                }
            }

            // Réenregistrer les crédences dans devi_prixcolori
            if ($request->has('kt_docs_repeater_credence') && is_array($request->kt_docs_repeater_credence)) {
                foreach ($request->kt_docs_repeater_credence as $plan) {
                    if (isset($plan['credence_longueur']) && isset($plan['credence_largeur']) && isset($plan['credence_epaisseur'])) {
                        $prixcolori = \App\Models\Prixcolori::find($plan['credence_epaisseur']);
                        $montant = 0;
                        if ($prixcolori) {
                            $surface = ($plan['credence_longueur'] / 1000) * ($plan['credence_largeur'] / 1000);
                            $montant = $surface * $prixcolori->prix;
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($request->addperc / 100));
                        }

                        \App\Models\DeviPrixcolori::create([
                            'devi_id' => $devi->id,
                            'longeur' => $plan['credence_longueur'],
                            'largeur' => $plan['credence_largeur'],
                            'prixcolori_id' => $plan['credence_epaisseur'],
                            'montant' => $montant,
                            'type' => 'credence'
                        ]);
                    }
                }
            }

            // Réenregistrer les options de découpe
            if ($request->has('optionsdecoupes') && is_array($request->optionsdecoupes)) {
                foreach ($request->optionsdecoupes as $optionsdecoupe_id => $quantite) {
                    if ($quantite > 0) {
                        // Récupérer le prix de l'option depuis la base de données
                        $optionsdecoupe = \App\Models\Optionsdecoupe::find($optionsdecoupe_id);
                        if ($optionsdecoupe) {
                            $prixu = $optionsdecoupe->prix;
                            $montant = $quantite * $prixu;
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($request->addperc / 100));

                            \App\Models\DeviOptionsdecoupe::create([
                                'devi_id' => $devi->id,
                                'optionsdecoupe_id' => $optionsdecoupe_id,
                                'quantite' => $quantite,
                                'prixu' => $prixu,
                                'montant' => $montant,
                            ]);
                        }
                    }
                }
            }

            // Réenregistrer les services
            if ($request->has('services') && is_array($request->services)) {
                foreach ($request->services as $service_id => $value) {
                    if ($value) {
                        $service = \App\Models\Service::find($service_id);
                        if ($service) {
                            $prixService = $service->prix * (1 + ($request->addperc / 100));
                            \App\Models\DeviService::create([
                                'devi_id' => $devi->id,
                                'service_id' => $service_id,
                                'devis_prix' => $prixService
                            ]);
                        }
                    }
                }
            }

            // Recalculer le montant HT total
            $totalHT = 0;

            // Ajouter les montants des plans de travail et crédences
            $prixcoloris = \App\Models\DeviPrixcolori::where('devi_id', $devi->id)->get();
            foreach ($prixcoloris as $prixcolori) {
                $totalHT += $prixcolori->montant;
            }

            // Ajouter les montants des options de découpe
            $optionsDecoupes = \App\Models\DeviOptionsdecoupe::where('devi_id', $devi->id)->get();
            foreach ($optionsDecoupes as $optionDecoupe) {
                $totalHT += $optionDecoupe->montant;
            }

            // Ajouter les prix des services
            $services = \App\Models\DeviService::where('devi_id', $devi->id)->get();
            foreach ($services as $service) {
                $totalHT += $service->devis_prix;
            }

            // Ajouter le prix de la bordure si elle a un prix
            if ($request->bordure_id && $request->bordure_id !== '0') {
                $bordure = \App\Models\Bordure::find($request->bordure_id);
                if ($bordure && $bordure->prix > 0) {
                    $montantBordure = floatval($bordure->prix);
                    // Appliquer le pourcentage addperc
                    $montantBordure = $montantBordure * (1 + ($request->addperc / 100));
                    $totalHT += $montantBordure;
                }
            }

            // Mettre à jour le devis avec le montant HT calculé
            $devi->update([
                'total_ht' => $totalHT
            ]);

            return redirect()->route('admin.devis.edit', $devi)->with('success', 'Devis modifié avec succès');
        } catch (\Exception $e) {
            //dd($e);
            return redirect()->back()->withInput()->with('error', 'Erreur lors de la modification: ' . $e->getMessage());
        }
    }

    public function destroy(Devi $devi)
    {
        $devi->delete();
        return redirect()->route('admin.devis.index')->with('success', 'Devis supprimé avec succès');
    }

    public function calculateTotal(Request $request)
    {
        try {
            // Décoder le JSON si nécessaire
            $data = $request->all();
            if ($request->header('Content-Type') === 'application/json') {
                $data = $request->json()->all();
            }

            $totalHT = 0;

            // Récupérer le pourcentage addperc
            $addperc = isset($data['addperc']) ? floatval($data['addperc']) : 0;

            // Calculer les plans de travail
            if (isset($data['kt_docs_repeater_plantravail']) && is_array($data['kt_docs_repeater_plantravail'])) {
                foreach ($data['kt_docs_repeater_plantravail'] as $plan) {
                    if (isset($plan['pl-longueur']) && isset($plan['pl-largeur']) && isset($plan['pl-epaisseur'])) {
                        $longueur = floatval($plan['pl-longueur']);
                        $largeur = floatval($plan['pl-largeur']);
                        $prixcolori = \App\Models\Prixcolori::find($plan['pl-epaisseur']);

                        if ($prixcolori && $longueur > 0 && $largeur > 0) {
                            // Conversion mm vers m²
                            $surface = ($longueur / 1000) * ($largeur / 1000);
                            $montant = $surface * floatval($prixcolori->prix);
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($addperc / 100));
                            $totalHT += $montant;

                            \Log::info("Plan de travail: {$longueur}x{$largeur}mm = {$surface}m² x {$prixcolori->prix}€ + {$addperc}% = {$montant}€");
                        }
                    }
                }
            }

            // Calculer les crédences
            if (isset($data['kt_docs_repeater_credence']) && is_array($data['kt_docs_repeater_credence'])) {
                foreach ($data['kt_docs_repeater_credence'] as $credence) {
                    if (isset($credence['credence_longueur']) && isset($credence['credence_largeur']) && isset($credence['credence_epaisseur'])) {
                        $longueur = floatval($credence['credence_longueur']);
                        $largeur = floatval($credence['credence_largeur']);
                        $prixcolori = \App\Models\Prixcolori::find($credence['credence_epaisseur']);

                        if ($prixcolori && $longueur > 0 && $largeur > 0) {
                            // Conversion mm vers m²
                            $surface = ($longueur / 1000) * ($largeur / 1000);
                            $montant = $surface * floatval($prixcolori->prix);
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($addperc / 100));
                            $totalHT += $montant;

                            \Log::info("Crédence: {$longueur}x{$largeur}mm = {$surface}m² x {$prixcolori->prix}€ + {$addperc}% = {$montant}€");
                        }
                    }
                }
            }

            // Calculer les options de découpe
            if (isset($data['optionsdecoupes']) && is_array($data['optionsdecoupes'])) {
                foreach ($data['optionsdecoupes'] as $option_id => $quantite) {
                    $quantite = floatval($quantite);
                    if ($quantite > 0) {
                        $option = \App\Models\Optionsdecoupe::find($option_id);
                        if ($option) {
                            $montant = $quantite * floatval($option->prix);
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($addperc / 100));
                            $totalHT += $montant;

                            \Log::info("Option découpe {$option->titre}: {$quantite} x {$option->prix}€ + {$addperc}% = {$montant}€");
                        }
                    }
                }
            }

            // Calculer les services
            if (isset($data['services']) && is_array($data['services'])) {
                foreach ($data['services'] as $service_id => $selected) {
                    if ($selected) {
                        $service = \App\Models\Service::find($service_id);
                        if ($service) {
                            $montant = floatval($service->prix);
                            // Appliquer le pourcentage addperc
                            $montant = $montant * (1 + ($addperc / 100));
                            $totalHT += $montant;

                            \Log::info("Service {$service->titre}: {$montant}€ + {$addperc}% = {$montant}€");
                        }
                    }
                }
            }

            // Calculer la bordure (si elle a un prix)
            if (isset($data['bordure_id']) && $data['bordure_id'] && $data['bordure_id'] !== '0') {
                $bordure = \App\Models\Bordure::find($data['bordure_id']);
                if ($bordure && $bordure->prix > 0) {
                    $montant = floatval($bordure->prix);
                    // Appliquer le pourcentage addperc
                    $montant = $montant * (1 + ($addperc / 100));
                    $totalHT += $montant;

                    \Log::info("Bordure {$bordure->titre}: {$bordure->prix}€ + {$addperc}% = {$montant}€");
                }
            }

            // Déterminer la TVA
            $tva = 20; // Valeur par défaut
            if (isset($data['tva'])) {
                if (is_bool($data['tva'])) {
                    $tva = $data['tva'] ? 10 : 20;
                } else {
                    $tva = intval($data['tva']);
                }
            }

            $totalTTC = $totalHT * (1 + ($tva / 100));

            // Récupérer le montant original pour comparaison (en mode édition)
            $montantOriginal = 0;
            if (isset($data['devis_id']) && $data['devis_id']) {
                $devis = Devi::find($data['devis_id']);
                if ($devis) {
                    $montantOriginal = floatval($devis->total_ht);
                }
            }

            \Log::info("Total HT calculé: {$totalHT}€, TVA: {$tva}%, Total TTC: {$totalTTC}€");

            return response()->json([
                'success' => true,
                'total_ht' => number_format($totalHT, 2, '.', ''),
                'total_ttc' => number_format($totalTTC, 2, '.', ''),
                'total_ht_formatted' => number_format($totalHT, 2, ',', ' ') . ' €',
                'total_ttc_formatted' => number_format($totalTTC, 2, ',', ' ') . ' €',
                'tva' => $tva,
                'montant_original' => $montantOriginal,
                'difference' => $totalHT - $montantOriginal
            ]);

        } catch (\Exception $e) {
            \Log::error('Erreur calcul total: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors du calcul: ' . $e->getMessage()
            ], 500);
        }
    }

    public function autoSave(Request $request)
    {
        try {
            // Validation des données minimales
            $request->validate([
                'client_id' => 'required|exists:clients,id',
                'matiere_id' => 'required|exists:matieres,id',
                'colori_id' => 'required|exists:coloris,id',
                'finition_id' => 'nullable|exists:finitions,id',
            ]);

            // Vérifier si un devis existe déjà pour ce client avec les mêmes caractéristiques
            $existingDevis = Devi::where('client_id', $request->client_id)
                ->where('matiere_id', $request->matiere_id)
                ->where('colori_id', $request->colori_id)
                ->where('statut', 0) // Brouillon
                ->first();

            if ($existingDevis) {
                // Mettre à jour le devis existant
                $existingDevis->update([
                    'finition_id' => $request->finition_id ?: $existingDevis->finition_id,
                ]);

                return response()->json([
                    'success' => true,
                    'devis_id' => $existingDevis->id,
                    'message' => 'Devis mis à jour automatiquement'
                ]);
            } else {
                // Créer un nouveau devis en brouillon
                $devis = Devi::create([
                    'client_id' => $request->client_id,
                    'matiere_id' => $request->matiere_id,
                    'colori_id' => $request->colori_id,
                    'finition_id' => $request->finition_id ?: 1, // Valeur par défaut
                    'statut' => 0, // Brouillon
                    'devis_pdf' => '', // Vide pour le moment
                    'total_ht' => 0,
                    'tva' => 0
                ]);

                return response()->json([
                    'success' => true,
                    'devis_id' => $devis->id,
                    'message' => 'Devis créé automatiquement'
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la sauvegarde: ' . $e->getMessage()
            ], 500);
        }
    }

    public function generatePdf(Devi $devi)
    {
        try {
            // Charger le devis avec toutes ses relations
            $devi->load([
                'client', 'matiere', 'colori', 'finition', 'bordure',
                'prixcoloris.prixcolori.epaisseur',
                'optionsdecoupes.optionsdecoupe',
                'services.service'
            ]);

            // Calculer le total TTC
            $totalTTC = $devi->total_ht * (1 + ($devi->tva / 100));

            // Préparer les données pour la vue PDF
            $data = [
                'devi' => $devi,
                'totalTTC' => $totalTTC,
                'dateGeneration' => Carbon::now()->format('d/m/Y'),
            ];

            // Générer le PDF avec des options spécifiques
            $pdf = Pdf::loadView('admin.devis.pdf-template', $data);
            $pdf->setPaper('A4', 'portrait');
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isPhpEnabled' => true,
                'chroot' => public_path(),
                'enable_remote' => false,
                'defaultFont' => 'sans-serif'
            ]);

            // Générer un nom de fichier unique
            $filename = 'devis_' . $devi->id . '_' . Carbon::now()->format('Ymd_His') . '.pdf';

            // Mettre à jour le champ devis_pdf avec le nom du fichier
            $devi->update(['devis_pdf' => $filename]);

            // Retourner le PDF en téléchargement
            return $pdf->download($filename);

        } catch (\Exception $e) {
            \Log::error('Erreur génération PDF: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return redirect()->back()->with('error', 'Erreur lors de la génération du PDF: ' . $e->getMessage());
        }
    }

    public function previewPdf(Devi $devi)
    {
        try {
            // Charger le devis avec toutes ses relations
            $devi->load([
                'client', 'matiere', 'colori', 'finition', 'bordure',
                'prixcoloris.prixcolori',
                'optionsdecoupes.optionsdecoupe',
                'services.service'
            ]);

            // Calculer le total TTC
            $totalTTC = $devi->total_ht * (1 + ($devi->tva / 100));

            // Préparer les données pour la vue PDF
            $data = [
                'devi' => $devi,
                'totalTTC' => $totalTTC,
                'dateGeneration' => Carbon::now()->format('d/m/Y'),
            ];

            // Générer le PDF et l'afficher dans le navigateur
            $pdf = Pdf::loadView('admin.devis.pdf-template', $data);
            $pdf->setPaper('A4', 'portrait');

            return $pdf->stream('preview_devis_' . $devi->id . '.pdf');

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erreur lors de la génération du PDF: ' . $e->getMessage());
        }
    }
}