<?php

namespace App\Http\Controllers;

use App\Models\Epaisseur;
use Illuminate\Http\Request;

class EpaisseurController extends Controller
{
    public function index()
    {
        $epaisseurs = Epaisseur::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.epaisseurs.index', compact('epaisseurs'));
    }

    public function create()
    {
        return view('admin.epaisseurs.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'boolean',
            'ordre' => 'integer'
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        // Obtenir le prochain ordre
        $maxOrdre = Epaisseur::max('ordre') ?? 0;
        
        Epaisseur::create([
            'titre' => $request->titre,
            'photo' => $photoPath,
            'pub' => $request->pub ?? 1,
            'ordre' => $request->ordre ?? ($maxOrdre + 1)
        ]);

        return redirect()->route('admin.epaisseurs.index')->with('success', 'Épaisseur créée avec succès');
    }

    public function edit(Epaisseur $epaisseur)
    {
        return view('admin.epaisseurs.form', compact('epaisseur'));
    }

    public function update(Request $request, Epaisseur $epaisseur)
    {
        $request->validate([
            'titre' => 'required',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'boolean',
            'ordre' => 'integer'
        ]);

        $photoPath = $epaisseur->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $epaisseur->update([
            'titre' => $request->titre,
            'photo' => $photoPath,
            'pub' => $request->pub ?? 1,
            'ordre' => $request->ordre ?? $epaisseur->ordre
        ]);

        return redirect()->route('admin.epaisseurs.edit', $epaisseur)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Epaisseur $epaisseur)
    {
        $epaisseur->delete();
        return redirect()->route('admin.epaisseurs.index')->with('success', 'Épaisseur supprimée avec succès');
    }

    public function moveUp(Epaisseur $epaisseur)
    {
        $previousEpaisseur = Epaisseur::where('ordre', '<', $epaisseur->ordre)
                                    ->orderBy('ordre', 'desc')
                                    ->first();

        if ($previousEpaisseur) {
            $tempOrdre = $epaisseur->ordre;
            $epaisseur->ordre = $previousEpaisseur->ordre;
            $previousEpaisseur->ordre = $tempOrdre;

            $epaisseur->save();
            $previousEpaisseur->save();
        }

        return redirect()->route('admin.epaisseurs.index')->with('success', 'Ordre modifié avec succès');
    }

    public function moveDown(Epaisseur $epaisseur)
    {
        $nextEpaisseur = Epaisseur::where('ordre', '>', $epaisseur->ordre)
                                 ->orderBy('ordre', 'asc')
                                 ->first();

        if ($nextEpaisseur) {
            $tempOrdre = $epaisseur->ordre;
            $epaisseur->ordre = $nextEpaisseur->ordre;
            $nextEpaisseur->ordre = $tempOrdre;

            $epaisseur->save();
            $nextEpaisseur->save();
        }

        return redirect()->route('admin.epaisseurs.index')->with('success', 'Ordre modifié avec succès');
    }

    public function togglePublication($id)
    {
        $epaisseur = Epaisseur::findOrFail($id);
        $epaisseur->pub = !$epaisseur->pub;
        $epaisseur->save();

        $status = $epaisseur->pub ? 'publié' : 'dépublié';
        return redirect()->route('admin.epaisseurs.index')->with('success', "Épaisseur {$status} avec succès");
    }

    public function getEpaisseursByColoriFinition($coloriId, $finitionId)
    {
        try {
            $epaisseurs = \App\Models\Prixcolori::with(['epaisseur', 'colori', 'finition'])
                ->where('colori_id', $coloriId)
                ->where('finition_id', $finitionId)
                ->whereHas('epaisseur', function($query) {
                    $query->where('pub', 1);
                })
                ->get()
                ->map(function ($prixcolori) {
                    return [
                        'id' => $prixcolori->epaisseur->id,
                        'titre' => $prixcolori->epaisseur->titre,
                        'prix' => number_format($prixcolori->prix, 2, ',', ' '),
                        'prix_raw' => $prixcolori->prix,
                        'prix_colori_id' => $prixcolori->id
                    ];
                });

            return response()->json($epaisseurs);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Erreur lors du chargement des épaisseurs'], 500);
        }
    }
}
