<?php

namespace App\Http\Controllers;

use App\Models\Finition;
use Illuminate\Http\Request;

class FinitionController extends Controller
{
    public function index()
    {
        $finitions = Finition::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.finitions.index', compact('finitions'));
    }

    public function create()
    {
        return view('admin.finitions.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        Finition::create([
            'titre' => $request->titre,
            'photo' => $photoPath,
            'pub' => $request->pub ?? 1,
            'ordre' => $request->ordre ?? 1
        ]);

        return redirect()->route('admin.finitions.index')->with('success', 'Finition créée avec succès');
    }

    public function edit(Finition $finition)
    {
        return view('admin.finitions.form', compact('finition'));
    }

    public function update(Request $request, Finition $finition)
    {
        $request->validate([
            'titre' => 'required',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        $photoPath = $finition->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $finition->update([
            'titre' => $request->titre,
            'photo' => $photoPath,
            'pub' => $request->pub ?? 1,
            'ordre' => $request->ordre ?? 1
        ]);

        return redirect()->route('admin.finitions.edit', $finition)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Finition $finition)
    {
        $finition->delete();
        return redirect()->route('admin.finitions.index')->with('success', 'Finition supprimée avec succès');
    }

    public function moveUp(Finition $finition)
    {
        $higherFinition = Finition::where('ordre', '>', $finition->ordre)->orderBy('ordre')->first();
        
        if ($higherFinition) {
            $tempOrdre = $finition->ordre;
            $finition->ordre = $higherFinition->ordre;
            $higherFinition->ordre = $tempOrdre;
            
            $finition->save();
            $higherFinition->save();
        }

        return redirect()->route('admin.finitions.index');
    }

    public function moveDown(Finition $finition)
    {
        $lowerFinition = Finition::where('ordre', '<', $finition->ordre)->orderBy('ordre', 'desc')->first();
        
        if ($lowerFinition) {
            $tempOrdre = $finition->ordre;
            $finition->ordre = $lowerFinition->ordre;
            $lowerFinition->ordre = $tempOrdre;
            
            $finition->save();
            $lowerFinition->save();
        }

        return redirect()->route('admin.finitions.index');
    }

    public function togglePublication($id)
    {
        $finition = Finition::findOrFail($id);
        $finition->pub = !$finition->pub;
        $finition->save();

        return redirect()->route('admin.finitions.index');
    }

    public function getFinitionsByColoris($coloriId)
    {
        try {
            \Log::info("Récupération des finitions pour le coloris ID: " . $coloriId);
            
            // Récupérer les finitions qui ont des prix pour ce coloris
            $finitions = Finition::whereHas('prixcoloris', function($query) use ($coloriId) {
                $query->where('colori_id', $coloriId);
            })
            ->where('pub', 1)
            ->orderBy('ordre', 'desc')
            ->select('id', 'titre')
            ->get();

            \Log::info("Finitions trouvées: " . $finitions->count());
            
            return response()->json($finitions);

        } catch (\Exception $e) {
            \Log::error("Erreur lors de la récupération des finitions: " . $e->getMessage());
            return response()->json(['error' => 'Erreur lors de la récupération des finitions'], 500);
        }
    }
}
