<?php

namespace App\Http\Controllers;

use App\Models\Matiere;
use App\Models\Categoriematiere;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class MatiereController extends Controller
{
    public function index()
    {
        $matieres = Matiere::with('categoriematiere')->orderBy('ordre', 'desc')->paginate(10);
        return view('admin.matieres.index', compact('matieres'));
    }

    public function create()
    {
        $categoriematieres = Categoriematiere::where('pub', 1)->orderBy('titre')->get();
        return view('admin.matieres.form', compact('categoriematieres'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'categoriematiere_id' => 'required|exists:categoriematieres,id',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $logoPath = null;
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('photos', 'public');
        }

        $matiere = Matiere::create([
            'titre' => $request->titre,
            'description' => $request->description,
            'categoriematiere_id' => $request->categoriematiere_id,
            'slug' => Str::slug($request->titre),
            'pub' => $request->pub,
            'photo' => $photoPath,
            'logo' => $logoPath,
            'ordre' => 0,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        $matiere->ordre = $matiere->id;
        $matiere->save();

        return redirect()->route('admin.matieres.index')->with('success', 'Matière créée avec succès');
    }

    public function edit(Matiere $matiere)
    {
        $categoriematieres = Categoriematiere::where('pub', 1)->orderBy('titre')->get();
        return view('admin.matieres.form', compact('matiere', 'categoriematieres'));
    }

    public function update(Request $request, Matiere $matiere)
    {
        $request->validate([
            'titre' => 'required',
            'categoriematiere_id' => 'required|exists:categoriematieres,id',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $photoPath = $matiere->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $logoPath = $matiere->logo;
        if ($request->hasFile('logo')) {
            $logoPath = $request->file('logo')->store('photos', 'public');
        }

        $matiere->update([
            'titre' => $request->titre,
            'description' => $request->description,
            'categoriematiere_id' => $request->categoriematiere_id,
            'slug' => Str::slug($request->titre),
            'pub' => $request->pub,
            'photo' => $photoPath,
            'logo' => $logoPath,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        return redirect()->route('admin.matieres.edit', $matiere)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Matiere $matiere)
    {
        $matiere->delete();
        return redirect()->route('admin.matieres.index')->with('success', 'Matière supprimée avec succès');
    }

    public function moveUp(Matiere $matiere)
    {
        $upperItem = Matiere::where('ordre', '>', $matiere->ordre)->orderBy('ordre', 'asc')->first();
        if ($upperItem) {
            $tempOrdre = $matiere->ordre;
            $matiere->ordre = $upperItem->ordre;
            $upperItem->ordre = $tempOrdre;
            $matiere->save();
            $upperItem->save();
        }
        return redirect()->route('admin.matieres.index')->with('success', 'Ordre modifié avec succès');
    }

    public function moveDown(Matiere $matiere)
    {
        $lowerItem = Matiere::where('ordre', '<', $matiere->ordre)->orderBy('ordre', 'desc')->first();
        if ($lowerItem) {
            $tempOrdre = $matiere->ordre;
            $matiere->ordre = $lowerItem->ordre;
            $lowerItem->ordre = $tempOrdre;
            $matiere->save();
            $lowerItem->save();
        }
        return redirect()->route('admin.matieres.index')->with('success', 'Ordre modifié avec succès');
    }

    public function togglePublication($id)
    {
        $matiere = Matiere::findOrFail($id);
        $matiere->pub = !$matiere->pub;
        $matiere->save();
        return redirect()->route('admin.matieres.index');
    }
}
