<?php

namespace App\Http\Controllers;

use App\Models\Nosprojet;
use Illuminate\Http\Request;

class NosprojetController extends Controller
{
    public function index()
    {
        $nosprojets = Nosprojet::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.nosprojets.index', compact('nosprojets'));
    }

    public function create()
    {
        return view('admin.nosprojets.form');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'pub' => 'required|in:0,1',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:10240',
                'ordre' => 'nullable|integer|min:1',
            ], [
                'titre.required' => 'Le titre est obligatoire.',
                'titre.max' => 'Le titre ne peut pas dépasser 255 caractères.',
                'pub.required' => 'Le statut de publication est obligatoire.',
                'pub.in' => 'Le statut de publication doit être 0 ou 1.',
                'photo.image' => 'Le fichier doit être une image.',
                'photo.mimes' => 'L\'image doit être au format JPEG, PNG ou JPG.',
                'photo.max' => 'L\'image ne peut pas dépasser 10 Mo.',
                'ordre.integer' => 'L\'ordre doit être un nombre entier.',
                'ordre.min' => 'L\'ordre doit être supérieur à 0.',
            ]);

            $photoPath = null;
            if ($request->hasFile('photo')) {
                $file = $request->file('photo');
                if ($file->isValid()) {
                    $photoPath = $file->store('photos', 'public');
                } else {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['photo' => 'Erreur lors de l\'upload du fichier: ' . $file->getErrorMessage()]);
                }
            }

            // Déterminer l'ordre pour le nouveau projet
            $ordre = $request->ordre;
            if (!$ordre) {
                $maxOrdre = Nosprojet::max('ordre');
                $ordre = $maxOrdre ? $maxOrdre + 1 : 1;
            } else {
                // Vérifier si l'ordre existe déjà et ajuster si nécessaire
                $existingProjet = Nosprojet::where('ordre', $ordre)->first();
                if ($existingProjet) {
                    // Décaler tous les projets avec un ordre >= à celui-ci
                    Nosprojet::where('ordre', '>=', $ordre)->increment('ordre');
                }
            }

            $nosprojet = Nosprojet::create([
                'titre' => $request->titre,
                'pub' => (bool)$request->pub,
                'photo' => $photoPath,
                'ordre' => $ordre,
            ]);

            return redirect()->route('admin.nosprojets.index')->with('success', 'Projet créé avec succès');

        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Une erreur est survenue lors de la création du projet : ' . $e->getMessage()]);
        }
    }

    public function edit(Nosprojet $nosprojet)
    {
        return view('admin.nosprojets.form', compact('nosprojet'));
    }

    public function update(Request $request, Nosprojet $nosprojet)
    {
        $request->validate([
            'titre' => 'required',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:10240',
            'ordre' => 'nullable|integer',
        ]);

        $photoPath = $nosprojet->photo;
        if ($request->hasFile('photo')) {
            $file = $request->file('photo');
            if ($file->isValid()) {
                $photoPath = $file->store('photos', 'public');
            } else {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['photo' => 'Erreur lors de l\'upload du fichier: ' . $file->getErrorMessage()]);
            }
        }

        $nosprojet->update([
            'titre' => $request->titre,
            'pub' => $request->pub,
            'photo' => $photoPath,
            'ordre' => $request->ordre ?? $nosprojet->ordre,
        ]);

        return redirect()->route('admin.nosprojets.edit', $nosprojet)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Nosprojet $nosprojet)
    {
        $nosprojet->delete();
        return redirect()->route('admin.nosprojets.index')->with('success', 'Projet supprimé avec succès');
    }

    public function moveUp(Nosprojet $nosprojet)
    {
        // Trouver le projet avec l'ordre immédiatement supérieur
        $upperProjet = Nosprojet::where('ordre', '>', $nosprojet->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperProjet) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $nosprojet->ordre;
            $nosprojet->ordre = $upperProjet->ordre;
            $upperProjet->ordre = $tempOrdre;

            $nosprojet->save();
            $upperProjet->save();
        }

        return redirect()->route('admin.nosprojets.index')->with('success', 'Projet déplacé vers le haut avec succès');
    }

    public function moveDown(Nosprojet $nosprojet)
    {
        // Trouver le projet avec l'ordre immédiatement inférieur
        $lowerProjet = Nosprojet::where('ordre', '<', $nosprojet->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerProjet) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $nosprojet->ordre;
            $nosprojet->ordre = $lowerProjet->ordre;
            $lowerProjet->ordre = $tempOrdre;

            $nosprojet->save();
            $lowerProjet->save();
        }

        return redirect()->route('admin.nosprojets.index')->with('success', 'Projet déplacé vers le bas avec succès');
    }

    public function togglePublication($id)
    {
        $nosprojet = Nosprojet::findOrFail($id);
        $nosprojet->pub = !$nosprojet->pub;
        $nosprojet->save();

        return redirect()->route('admin.nosprojets.index');
    }
}
