<?php

namespace App\Http\Controllers;

use App\Models\Optionsdecoupe;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class OptionsdecoupeController extends Controller
{
    public function index()
    {
        $optionsdecoupes = Optionsdecoupe::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.optionsdecoupes.index', compact('optionsdecoupes'));
    }

    public function create()
    {
        return view('admin.optionsdecoupes.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'prix' => 'required|numeric|min:0',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'pub' => 'required|in:0,1',
            'ordre' => 'nullable|integer'
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $optionsdecoupe = Optionsdecoupe::create([
            'titre' => $request->titre,
            'prix' => $request->prix,
            'photo' => $photoPath,
            'pub' => $request->pub,
            'ordre' => 0, // Temporaire
        ]);

        // Mettre à jour le champ ordre avec l'ID
        $optionsdecoupe->ordre = $request->ordre ?? $optionsdecoupe->id;
        $optionsdecoupe->save();

        return redirect()->route('admin.optionsdecoupes.index')->with('success', 'Option de découpe créée avec succès');
    }

    public function edit(Optionsdecoupe $optionsdecoupe)
    {
        return view('admin.optionsdecoupes.form', compact('optionsdecoupe'));
    }

    public function update(Request $request, Optionsdecoupe $optionsdecoupe)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'prix' => 'required|numeric|min:0',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'pub' => 'required|in:0,1',
            'ordre' => 'nullable|integer'
        ]);

        $photoPath = $optionsdecoupe->photo;
        if ($request->hasFile('photo')) {
            if ($optionsdecoupe->photo) {
                Storage::disk('public')->delete($optionsdecoupe->photo);
            }
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $optionsdecoupe->update([
            'titre' => $request->titre,
            'prix' => $request->prix,
            'photo' => $photoPath,
            'pub' => $request->pub,
            'ordre' => $request->ordre ?? $optionsdecoupe->ordre,
        ]);

        return redirect()->route('admin.optionsdecoupes.edit', $optionsdecoupe)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Optionsdecoupe $optionsdecoupe)
    {
        if ($optionsdecoupe->photo) {
            Storage::disk('public')->delete($optionsdecoupe->photo);
        }

        $optionsdecoupe->delete();

        return redirect()->route('admin.optionsdecoupes.index')->with('success', 'Option de découpe supprimée avec succès');
    }

    public function moveUp(Optionsdecoupe $optionsdecoupe)
    {
        // Trouver l'option avec l'ordre immédiatement supérieur
        $upperOption = Optionsdecoupe::where('ordre', '>', $optionsdecoupe->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperOption) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $optionsdecoupe->ordre;
            $optionsdecoupe->ordre = $upperOption->ordre;
            $upperOption->ordre = $tempOrdre;

            $optionsdecoupe->save();
            $upperOption->save();
        }

        return redirect()->route('admin.optionsdecoupes.index')->with('success', 'Option déplacée vers le haut avec succès');
    }

    public function moveDown(Optionsdecoupe $optionsdecoupe)
    {
        // Trouver l'option avec l'ordre immédiatement inférieur
        $lowerOption = Optionsdecoupe::where('ordre', '<', $optionsdecoupe->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerOption) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $optionsdecoupe->ordre;
            $optionsdecoupe->ordre = $lowerOption->ordre;
            $lowerOption->ordre = $tempOrdre;

            $optionsdecoupe->save();
            $lowerOption->save();
        }

        return redirect()->route('admin.optionsdecoupes.index')->with('success', 'Option déplacée vers le bas avec succès');
    }

    public function togglePublication(Optionsdecoupe $optionsdecoupe)
    {
        $optionsdecoupe->pub = !$optionsdecoupe->pub;
        $optionsdecoupe->save();

        return redirect()->route('admin.optionsdecoupes.index');
    }

    /**
     * Get all published options de découpe for AJAX requests
     */
    public function getAllOptionsdecoupes()
    {
        try {
            $optionsdecoupes = Optionsdecoupe::where('pub', 1)
                ->orderBy('ordre', 'asc')
                ->get()
                ->map(function ($option) {
                    return [
                        'id' => $option->id,
                        'titre' => $option->titre,
                        'prix' => number_format($option->prix, 2),
                        'prix_raw' => $option->prix,
                        'photo' => $option->photo ? asset('storage/' . $option->photo) : null
                    ];
                });

            return response()->json($optionsdecoupes);
        } catch (\Exception $e) {
            \Log::error('Erreur getAllOptionsdecoupes: ' . $e->getMessage());
            return response()->json(['error' => 'Erreur lors du chargement des options de découpe'], 500);
        }
    }
}