<?php

namespace App\Http\Controllers;

use App\Models\Postcategory;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PostcategoryController extends Controller
{
    public function index()
    {
        $categories = Postcategory::orderBy('ordre', 'desc')->paginate(10);
        return view('postcategories.index', compact('categories'));
    }

    public function create()
    {
        return view('postcategories.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'ordre' => 'nullable|integer',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $photoPath = null;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $category = Postcategory::create([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'pub' => $request->pub,
            'photo' => $photoPath,
            'ordre' => 0, // Temporaire
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        // Mettre à jour le champ ordre avec l'ID
        $category->ordre = $request->ordre ?? $category->id;
        $category->save();

        return redirect()->route('postcategories.index')->with('success', 'Category created successfully');
    }

    public function edit(Postcategory $postcategory)
    {
        return view('postcategories.form', compact('postcategory'));
    }

    public function update(Request $request, Postcategory $postcategory)
    {
        $request->validate([
            'title' => 'required',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'ordre' => 'nullable|integer',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string'
        ]);

        $photoPath = $postcategory->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $postcategory->update([
            'title' => $request->title,
            'slug' => Str::slug($request->title),
            'pub' => $request->pub,
            'photo' => $photoPath,
            'ordre' => $request->ordre ?? $postcategory->ordre,
            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description
        ]);

        return redirect()->route('postcategories.edit', $postcategory)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Postcategory $postcategory)
    {
        $postcategory->delete();
        return redirect()->route('postcategories.index')->with('success', 'Category deleted successfully');
    }

    public function moveUp(Postcategory $postcategory)
    {
        // Trouver la catégorie avec l'ordre immédiatement supérieur
        $upperCategory = Postcategory::where('ordre', '>', $postcategory->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperCategory) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $postcategory->ordre;
            $postcategory->ordre = $upperCategory->ordre;
            $upperCategory->ordre = $tempOrdre;

            $postcategory->save();
            $upperCategory->save();
        }

        return redirect()->route('postcategories.index')->with('success', 'Category moved up successfully');
    }

    public function moveDown(Postcategory $postcategory)
    {
        // Trouver la catégorie avec l'ordre immédiatement inférieur
        $lowerCategory = Postcategory::where('ordre', '<', $postcategory->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerCategory) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $postcategory->ordre;
            $postcategory->ordre = $lowerCategory->ordre;
            $lowerCategory->ordre = $tempOrdre;

            $postcategory->save();
            $lowerCategory->save();
        }

        return redirect()->route('postcategories.index')->with('success', 'Category moved down successfully');
    }

    public function togglePublication($id)
    {
        $category = Postcategory::findOrFail($id);
        $category->pub = !$category->pub;
        $category->save();

        return redirect()->route('postcategories.index');
    }
}