<?php

namespace App\Http\Controllers;

use App\Models\Prixcolori;
use App\Models\Colori;
use App\Models\Finition;
use App\Models\Epaisseur;
use Illuminate\Http\Request;

class PrixcoloriController extends Controller
{
    public function index()
    {
        $prixcoloris = Prixcolori::with(['colori.matiere', 'finition', 'epaisseur'])->paginate(15);
        return view('admin.prixcoloris.index', compact('prixcoloris'));
    }

    public function create()
    {
        $coloris = Colori::with('matiere')->orderBy('titre')->get();
        $finitions = Finition::orderBy('titre')->get();
        $epaisseurs = Epaisseur::orderBy('titre')->get();
        return view('admin.prixcoloris.form', compact('coloris', 'finitions', 'epaisseurs'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'colori_id' => 'required|exists:coloris,id',
            'finition_id' => 'required|exists:finitions,id',
            'epaisseur_id' => 'required|exists:epaisseurs,id',
            'prix' => 'required|numeric|min:0'
        ], [
            'colori_id.required' => 'Veuillez sélectionner un coloris',
            'finition_id.required' => 'Veuillez sélectionner une finition',
            'epaisseur_id.required' => 'Veuillez sélectionner une épaisseur',
            'prix.required' => 'Le prix est obligatoire',
            'prix.numeric' => 'Le prix doit être un nombre',
            'prix.min' => 'Le prix doit être positif'
        ]);

        // Vérifier si cette combinaison existe déjà
        $exists = Prixcolori::where('colori_id', $request->colori_id)
            ->where('finition_id', $request->finition_id)
            ->where('epaisseur_id', $request->epaisseur_id)
            ->exists();

        if ($exists) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Cette combinaison coloris/finition/épaisseur existe déjà.']);
        }

        Prixcolori::create($request->all());

        return redirect()->route('admin.prixcoloris.index')->with('success', 'Tarification créée avec succès');
    }

    public function show(Prixcolori $prixcolori)
    {
        return view('admin.prixcoloris.show', compact('prixcolori'));
    }

    public function edit(Prixcolori $prixcolori)
    {
        $coloris = Colori::with('matiere')->orderBy('titre')->get();
        $finitions = Finition::orderBy('titre')->get();
        $epaisseurs = Epaisseur::orderBy('titre')->get();
        return view('admin.prixcoloris.form', compact('prixcolori', 'coloris', 'finitions', 'epaisseurs'));
    }

    public function update(Request $request, Prixcolori $prixcolori)
    {
        $request->validate([
            'colori_id' => 'required|exists:coloris,id',
            'finition_id' => 'required|exists:finitions,id',
            'epaisseur_id' => 'required|exists:epaisseurs,id',
            'prix' => 'required|numeric|min:0'
        ], [
            'colori_id.required' => 'Veuillez sélectionner un coloris',
            'finition_id.required' => 'Veuillez sélectionner une finition',
            'epaisseur_id.required' => 'Veuillez sélectionner une épaisseur',
            'prix.required' => 'Le prix est obligatoire',
            'prix.numeric' => 'Le prix doit être un nombre',
            'prix.min' => 'Le prix doit être positif'
        ]);

        // Vérifier si cette combinaison existe déjà (en excluant l'enregistrement actuel)
        $exists = Prixcolori::where('colori_id', $request->colori_id)
            ->where('finition_id', $request->finition_id)
            ->where('epaisseur_id', $request->epaisseur_id)
            ->where('id', '!=', $prixcolori->id)
            ->exists();

        if ($exists) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Cette combinaison coloris/finition/épaisseur existe déjà.']);
        }

        $prixcolori->update($request->all());

        return redirect()->route('admin.prixcoloris.index')->with('success', 'Tarification modifiée avec succès');
    }

    public function destroy(Prixcolori $prixcolori)
    {
        try {
            $prixcolori->delete();

            if (request()->ajax()) {
                return response()->json(['success' => true, 'message' => 'Tarification supprimée avec succès']);
            }

            return redirect()->route('admin.prixcoloris.index')->with('success', 'Tarification supprimée avec succès');
        } catch (\Exception $e) {
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => 'Erreur lors de la suppression']);
            }

            return redirect()->route('admin.prixcoloris.index')->with('error', 'Erreur lors de la suppression de la tarification');
        }
    }

    public function moveUp(Prixcolori $prixcolori)
    {
        // Implémentation pour déplacer vers le haut si nécessaire
        return redirect()->route('admin.prixcoloris.index');
    }

    public function moveDown(Prixcolori $prixcolori)
    {
        // Implémentation pour déplacer vers le bas si nécessaire
        return redirect()->route('admin.prixcoloris.index');
    }

    public function togglePublication($id)
    {
        // Implémentation pour toggle publication si nécessaire
        return redirect()->route('admin.prixcoloris.index');
    }
}