<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\Permission;
use Illuminate\Http\Request;

class RoleController extends Controller
{
    public function index()
    {
        $roles = Role::with('permissions')->orderBy('ordre')->paginate(15);
        return view('admin.roles.index', compact('roles'));
    }

    public function create()
    {
        $permissions = Permission::active()->orderBy('category')->orderBy('display_name')->get();
        $permissionsByCategory = $permissions->groupBy('category');
        return view('admin.roles.form', compact('permissionsByCategory'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:roles'],
            'display_name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'ordre' => ['required', 'integer', 'min:0'],
            'permissions' => ['array'],
            'permissions.*' => ['exists:permissions,id'],
        ]);

        $role = Role::create([
            'name' => $request->name,
            'display_name' => $request->display_name,
            'description' => $request->description,
            'ordre' => $request->ordre,
            'is_active' => $request->has('is_active'),
        ]);

        if ($request->permissions) {
            $role->permissions()->attach($request->permissions);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Rôle créé avec succès.');
    }

    public function show(Role $role)
    {
        $role->load('permissions', 'users');
        return view('admin.roles.show', compact('role'));
    }

    public function edit(Role $role)
    {
        $permissions = Permission::active()->orderBy('category')->orderBy('display_name')->get();
        $permissionsByCategory = $permissions->groupBy('category');
        $rolePermissions = $role->permissions->pluck('id')->toArray();
        
        return view('admin.roles.form', compact('role', 'permissionsByCategory', 'rolePermissions'));
    }

    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', 'unique:roles,name,' . $role->id],
            'display_name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'ordre' => ['required', 'integer', 'min:0'],
            'permissions' => ['array'],
            'permissions.*' => ['exists:permissions,id'],
        ]);

        $role->update([
            'name' => $request->name,
            'display_name' => $request->display_name,
            'description' => $request->description,
            'ordre' => $request->ordre,
            'is_active' => $request->has('is_active'),
        ]);

        $role->permissions()->sync($request->permissions ?? []);

        return redirect()->route('admin.roles.index')->with('success', 'Rôle modifié avec succès.');
    }

    public function destroy(Role $role)
    {
        if ($role->users()->count() > 0) {
            return redirect()->route('admin.roles.index')->with('error', 'Impossible de supprimer un rôle assigné à des utilisateurs.');
        }

        $role->permissions()->detach();
        $role->delete();

        return redirect()->route('admin.roles.index')->with('success', 'Rôle supprimé avec succès.');
    }

    public function toggleStatus(Role $role)
    {
        $role->update(['is_active' => !$role->is_active]);
        $status = $role->is_active ? 'activé' : 'désactivé';
        
        return redirect()->route('admin.roles.index')->with('success', "Rôle {$status} avec succès.");
    }

    public function moveUp(Role $role)
    {
        $previousRole = Role::where('ordre', '<', $role->ordre)
                            ->orderBy('ordre', 'desc')
                            ->first();

        if ($previousRole) {
            $tempOrdre = $role->ordre;
            $role->update(['ordre' => $previousRole->ordre]);
            $previousRole->update(['ordre' => $tempOrdre]);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Ordre modifié avec succès.');
    }

    public function moveDown(Role $role)
    {
        $nextRole = Role::where('ordre', '>', $role->ordre)
                        ->orderBy('ordre', 'asc')
                        ->first();

        if ($nextRole) {
            $tempOrdre = $role->ordre;
            $role->update(['ordre' => $nextRole->ordre]);
            $nextRole->update(['ordre' => $tempOrdre]);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Ordre modifié avec succès.');
    }
}
