<?php

namespace App\Http\Controllers;

use App\Models\Service;
use Illuminate\Http\Request;

class ServiceController extends Controller
{
    public function index()
    {
        $services = Service::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.services.index', compact('services'));
    }

    public function create()
    {
        return view('admin.services.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'prix' => 'required|numeric|min:0',
            'pub' => 'boolean'
        ]);

        $data = $request->all();
        $data['pub'] = $request->has('pub') ? 1 : 0;
        
        // Assigner l'ordre le plus élevé + 1
        $data['ordre'] = Service::max('ordre') + 1;

        Service::create($data);

        return redirect()->route('admin.services.index')->with('success', 'Service créé avec succès');
    }

    public function edit(Service $service)
    {
        return view('admin.services.form', compact('service'));
    }

    public function update(Request $request, Service $service)
    {
        $request->validate([
            'titre' => 'required|string|max:255',
            'prix' => 'required|numeric|min:0',
            'pub' => 'boolean'
        ]);

        $data = $request->all();
        $data['pub'] = $request->has('pub') ? 1 : 0;

        $service->update($data);

        return redirect()->route('admin.services.index')->with('success', 'Service modifié avec succès');
    }

    public function destroy(Service $service)
    {
        $service->delete();
        return redirect()->route('admin.services.index')->with('success', 'Service supprimé avec succès');
    }

    public function moveUp(Service $service)
    {
        $nextService = Service::where('ordre', '>', $service->ordre)->orderBy('ordre')->first();
        
        if ($nextService) {
            $currentOrdre = $service->ordre;
            $service->ordre = $nextService->ordre;
            $nextService->ordre = $currentOrdre;
            
            $service->save();
            $nextService->save();
        }
        
        return redirect()->route('admin.services.index')->with('success', 'Service déplacé vers le haut avec succès');
    }

    public function moveDown(Service $service)
    {
        $previousService = Service::where('ordre', '<', $service->ordre)->orderBy('ordre', 'desc')->first();
        
        if ($previousService) {
            $currentOrdre = $service->ordre;
            $service->ordre = $previousService->ordre;
            $previousService->ordre = $currentOrdre;
            
            $service->save();
            $previousService->save();
        }
        
        return redirect()->route('admin.services.index')->with('success', 'Service déplacé vers le bas avec succès');
    }

    public function togglePublication(Service $service)
    {
        $service->pub = !$service->pub;
        $service->save();

        $message = $service->pub ? 'Service publié avec succès' : 'Service dépublié avec succès';
        return redirect()->route('admin.services.index')->with('success', $message);
    }
}
