<?php

namespace App\Http\Controllers;

use App\Models\Slide;
use Illuminate\Http\Request;

class SlideController extends Controller
{
    public function index()
    {
        $slides = Slide::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.slides.index', compact('slides'));
    }

    public function create()
    {
        return view('admin.slides.form');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'titre' => 'required|string|max:255',
                'description' => 'nullable|string|max:1000',
                'lien' => 'nullable|url|max:255',
                'pub' => 'required|in:0,1',
                'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:10240',
                'ordre' => 'nullable|integer|min:1',
            ], [
                'titre.required' => 'Le titre est obligatoire.',
                'titre.max' => 'Le titre ne peut pas dépasser 255 caractères.',
                'description.max' => 'La description ne peut pas dépasser 1000 caractères.',
                'lien.url' => 'Le lien doit être une URL valide.',
                'lien.max' => 'Le lien ne peut pas dépasser 255 caractères.',
                'pub.required' => 'Le statut de publication est obligatoire.',
                'pub.in' => 'Le statut de publication doit être 0 ou 1.',
                'photo.image' => 'Le fichier doit être une image.',
                'photo.mimes' => 'L\'image doit être au format JPEG, PNG ou JPG.',
                'photo.max' => 'L\'image ne peut pas dépasser 10 Mo.',
                'ordre.integer' => 'L\'ordre doit être un nombre entier.',
                'ordre.min' => 'L\'ordre doit être supérieur à 0.',
            ]);

            $photoPath = null;
            if ($request->hasFile('photo')) {
                $file = $request->file('photo');
                if ($file->isValid()) {
                    $photoPath = $file->store('photos', 'public');
                } else {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['photo' => 'Erreur lors de l\'upload du fichier: ' . $file->getErrorMessage()]);
                }
            }

            // Déterminer l'ordre pour le nouveau slide
            $ordre = $request->ordre;
            if (!$ordre) {
                $maxOrdre = Slide::max('ordre');
                $ordre = $maxOrdre ? $maxOrdre + 1 : 1;
            } else {
                // Vérifier si l'ordre existe déjà et ajuster si nécessaire
                $existingSlide = Slide::where('ordre', $ordre)->first();
                if ($existingSlide) {
                    // Décaler tous les slides avec un ordre >= à celui-ci
                    Slide::where('ordre', '>=', $ordre)->increment('ordre');
                }
            }

            $slide = Slide::create([
                'titre' => $request->titre,
                'description' => $request->description,
                'lien' => $request->lien,
                'pub' => (bool)$request->pub,
                'photo' => $photoPath,
                'ordre' => $ordre,
            ]);

            return redirect()->route('admin.slides.index')->with('success', 'Slide créé avec succès');

        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Une erreur est survenue lors de la création du slide : ' . $e->getMessage()]);
        }
    }

    public function edit(Slide $slide)
    {
        return view('admin.slides.form', compact('slide'));
    }

    public function update(Request $request, Slide $slide)
    {
        $request->validate([
            'titre' => 'required',
            'description' => 'nullable|string',
            'lien' => 'nullable|url|max:255',
            'pub' => 'required|in:0,1',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:10240',
            'ordre' => 'nullable|integer',
        ]);

        $photoPath = $slide->photo;
        if ($request->hasFile('photo')) {
            $file = $request->file('photo');
            if ($file->isValid()) {
                $photoPath = $file->store('photos', 'public');
            } else {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['photo' => 'Erreur lors de l\'upload du fichier: ' . $file->getErrorMessage()]);
            }
        }

        $slide->update([
            'titre' => $request->titre,
            'description' => $request->description,
            'lien' => $request->lien,
            'pub' => $request->pub,
            'photo' => $photoPath,
            'ordre' => $request->ordre ?? $slide->ordre,
        ]);

        return redirect()->route('admin.slides.edit', $slide)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Slide $slide)
    {
        $slide->delete();
        return redirect()->route('admin.slides.index')->with('success', 'Slide supprimé avec succès');
    }

    public function moveUp(Slide $slide)
    {
        // Trouver le slide avec l'ordre immédiatement supérieur
        $upperSlide = Slide::where('ordre', '>', $slide->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperSlide) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $slide->ordre;
            $slide->ordre = $upperSlide->ordre;
            $upperSlide->ordre = $tempOrdre;

            $slide->save();
            $upperSlide->save();
        }

        return redirect()->route('admin.slides.index')->with('success', 'Slide déplacé vers le haut avec succès');
    }

    public function moveDown(Slide $slide)
    {
        // Trouver le slide avec l'ordre immédiatement inférieur
        $lowerSlide = Slide::where('ordre', '<', $slide->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerSlide) {
            // Échanger les valeurs d'ordre
            $tempOrdre = $slide->ordre;
            $slide->ordre = $lowerSlide->ordre;
            $lowerSlide->ordre = $tempOrdre;

            $slide->save();
            $lowerSlide->save();
        }

        return redirect()->route('admin.slides.index')->with('success', 'Slide déplacé vers le bas avec succès');
    }

    public function togglePublication($id)
    {
        $slide = Slide::findOrFail($id);
        $slide->pub = !$slide->pub;
        $slide->save();

        return redirect()->route('admin.slides.index');
    }
}