<?php

namespace App\Http\Controllers;

use App\Models\Temoinage;
use Illuminate\Http\Request;

class TemoinageController extends Controller
{
    public function index()
    {
        $temoinages = Temoinage::orderBy('ordre', 'desc')->paginate(10);
        return view('admin.temoinages.index', compact('temoinages'));
    }

    public function create()
    {
        return view('admin.temoinages.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'titre' => 'required',
            'nom' => 'required',
            'description' => 'required',
            'photo' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'required|in:0,1'
        ]);

        $photoPath = $request->file('photo')->store('photos', 'public');

        $temoinage = Temoinage::create([
            'titre' => $request->titre,
            'nom' => $request->nom,
            'description' => $request->description,
            'photo' => $photoPath,
            'pub' => $request->pub,
            'ordre' => 0
        ]);

        $temoinage->ordre = $temoinage->id;
        $temoinage->save();

        return redirect()->route('admin.temoinages.index')->with('success', 'Témoignage créé avec succès');
    }

    public function edit(Temoinage $temoinage)
    {
        return view('admin.temoinages.form', compact('temoinage'));
    }

    public function update(Request $request, Temoinage $temoinage)
    {
        $request->validate([
            'titre' => 'required',
            'nom' => 'required',
            'description' => 'required',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'pub' => 'required|in:0,1'
        ]);

        $photoPath = $temoinage->photo;
        if ($request->hasFile('photo')) {
            $photoPath = $request->file('photo')->store('photos', 'public');
        }

        $temoinage->update([
            'titre' => $request->titre,
            'nom' => $request->nom,
            'description' => $request->description,
            'photo' => $photoPath,
            'pub' => $request->pub
        ]);

        return redirect()->route('admin.temoinages.edit', $temoinage)->with('success', 'Sauvegarde effectuée avec succès');
    }

    public function destroy(Temoinage $temoinage)
    {
        $temoinage->delete();
        return redirect()->route('admin.temoinages.index')->with('success', 'Témoignage supprimé avec succès');
    }

    public function moveUp(Temoinage $temoinage)
    {
        $upperTemoinage = Temoinage::where('ordre', '>', $temoinage->ordre)
            ->orderBy('ordre', 'asc')
            ->first();

        if ($upperTemoinage) {
            $tempOrdre = $temoinage->ordre;
            $temoinage->ordre = $upperTemoinage->ordre;
            $upperTemoinage->ordre = $tempOrdre;

            $temoinage->save();
            $upperTemoinage->save();
        }

        return redirect()->route('admin.temoinages.index')->with('success', 'Témoignage déplacé vers le haut');
    }

    public function moveDown(Temoinage $temoinage)
    {
        $lowerTemoinage = Temoinage::where('ordre', '<', $temoinage->ordre)
            ->orderBy('ordre', 'desc')
            ->first();

        if ($lowerTemoinage) {
            $tempOrdre = $temoinage->ordre;
            $temoinage->ordre = $lowerTemoinage->ordre;
            $lowerTemoinage->ordre = $tempOrdre;

            $temoinage->save();
            $lowerTemoinage->save();
        }

        return redirect()->route('admin.temoinages.index')->with('success', 'Témoignage déplacé vers le bas');
    }

    public function togglePublication($id)
    {
        $temoinage = Temoinage::findOrFail($id);
        $temoinage->pub = !$temoinage->pub;
        $temoinage->save();

        return redirect()->route('admin.temoinages.index');
    }
}
