// Variable globale pour stocker les épaisseurs actuelles
window.currentEpaisseurs = [];

// Fonction pour peupler un select d'épaisseur
function populateEpaisseurSelect(select) {
    select.innerHTML = '<option value="">Choisissez une épaisseur</option>';

    if (window.currentEpaisseurs.length > 0) {
        window.currentEpaisseurs.forEach((epaisseur) => {
            const option = document.createElement("option");
            option.value = epaisseur.id;
            option.textContent = `${epaisseur.titre} - ${epaisseur.prix}€/m²`;
            option.setAttribute(
                "data-prix",
                epaisseur.prix_raw || epaisseur.prix,
            );
            option.setAttribute(
                "data-prix-colori-id",
                epaisseur.prix_colori_id,
            );
            select.appendChild(option);
        });
    }

    // Réinitialiser Select2 pour ce select
    if ($(select).hasClass("select2-hidden-accessible")) {
        $(select).select2("destroy");
    }
    $(select).select2({
        minimumResultsForSearch: Infinity,
    });

    // Attacher les événements de calcul pour ce select spécifique
    ["change", "input"].forEach((event) => {
        select.removeEventListener(event, calculateTotal);
        select.addEventListener(event, calculateTotal);
    });
}

// Fonction consolidée pour calculer le total et mettre à jour le récapitulatif
function calculateTotal() {
    let total = 0;
    let recapitulatif = {
        etape1: { titre: "Informations Client", data: {} },
        etape2: { titre: "Sélection Matière", data: {} },
        etape3: {
            titre: "Plans de travail",
            data: [],
            totalArea: 0,
            totalCost: 0,
        },
        etape4: { titre: "Crédences", data: [], totalArea: 0, totalCost: 0 },
        etape5: { titre: "Bordures", data: {} },
        etape6: { titre: "Options de découpe", data: [], totalCost: 0 },
        etape7: { titre: "Services", data: [], totalCost: 0 },
        etape8: { titre: "Accessoires", data: [], totalCost: 0 },
    };

    // ÉTAPE 1 - Informations Client
    const nomInput = document.querySelector(
        'input[name="nom"], input[name="onest-nom"]',
    );
    const prenomInput = document.querySelector(
        'input[name="prenom"], input[name="onest-prenom"]',
    );
    const emailInput = document.querySelector(
        'input[name="email"], input[name="onest-email"]',
    );
    const telephoneInput = document.querySelector(
        'input[name="telephone"], input[name="onest-tel"]',
    );
    const adresseInput = document.querySelector(
        'textarea[name="adresse"], textarea[name="description"]',
    );

    if (nomInput?.value || prenomInput?.value || emailInput?.value) {
        recapitulatif.etape1.data = {
            nom: nomInput?.value || "",
            prenom: prenomInput?.value || "",
            email: emailInput?.value || "",
            telephone: telephoneInput?.value || "",
            adresse: adresseInput?.value || "",
        };
    }

    // ÉTAPE 2 - Sélection Matière
    const matiereSelect = document.querySelector("#matiere_id");
    const coloriSelect = document.querySelector("#colori_id");
    const finitionSelect = document.querySelector('select[name="finition_id"]');

    if (matiereSelect?.value) {
        recapitulatif.etape2.data = {
            matiere:
                matiereSelect.options[matiereSelect.selectedIndex]?.text || "",
            colori:
                coloriSelect?.options[coloriSelect.selectedIndex]?.text || "",
            finition:
                finitionSelect?.options[finitionSelect.selectedIndex]?.text ||
                "",
        };
    }

    // ÉTAPE 3 - Plans de travail
    document
        .querySelectorAll("#kt_docs_repeater_basic [data-repeater-item]")
        .forEach((item, index) => {
            const longueurInput = item.querySelector('input[name*="longueur"]');
            const largeurInput = item.querySelector('input[name*="largeur"]');
            const epaisseurSelect = item.querySelector(
                'select[name*="epaisseur"]',
            );

            if (!longueurInput || !largeurInput || !epaisseurSelect) return;

            const longueur = parseFloat(longueurInput.value) || 0;
            const largeur = parseFloat(largeurInput.value) || 0;
            const epaisseurId = epaisseurSelect.value;
            let prixMetreCarre = 0;
            let epaisseurText = "";

            if (window.currentEpaisseurs && epaisseurId) {
                const epaisseur = window.currentEpaisseurs.find(
                    (ep) => ep.id == epaisseurId,
                );
                if (epaisseur) {
                    prixMetreCarre = parseFloat(epaisseur.prix) || 0;
                    epaisseurText = epaisseur.titre || "";
                }
            }

            if (longueur > 0 && largeur > 0) {
                const area = (longueur * largeur) / 1000000;
                const itemCost = area * prixMetreCarre;

                recapitulatif.etape3.data.push({
                    index: index + 1,
                    longueur: longueur,
                    largeur: largeur,
                    epaisseur: epaisseurText,
                    area: area,
                    prixUnitaire: prixMetreCarre,
                    cost: itemCost,
                });

                recapitulatif.etape3.totalArea += area;
                recapitulatif.etape3.totalCost += itemCost;
                total += itemCost;
            }
        });

    // ÉTAPE 4 - Crédences
    document
        .querySelectorAll("#kt_docs_repeater_credence [data-repeater-item]")
        .forEach((item, index) => {
            const longueurInput = item.querySelector('input[name*="longueur"]');
            const largeurInput = item.querySelector('input[name*="largeur"]');
            const epaisseurSelect = item.querySelector(
                'select[name*="epaisseur"]',
            );

            if (!longueurInput || !largeurInput || !epaisseurSelect) return;

            const longueur = parseFloat(longueurInput.value) || 0;
            const largeur = parseFloat(largeurInput.value) || 0;
            const epaisseurId = epaisseurSelect.value;
            let prixMetreCarre = 0;
            let epaisseurText = "";

            if (window.currentEpaisseurs && epaisseurId) {
                const epaisseur = window.currentEpaisseurs.find(
                    (ep) => ep.id == epaisseurId,
                );
                if (epaisseur) {
                    prixMetreCarre = parseFloat(epaisseur.prix) || 0;
                    epaisseurText = epaisseur.titre || "";
                }
            }

            if (longueur > 0 && largeur > 0) {
                const area = (longueur * largeur) / 1000000;
                const itemCost = area * prixMetreCarre;

                recapitulatif.etape4.data.push({
                    index: index + 1,
                    longueur: longueur,
                    largeur: largeur,
                    epaisseur: epaisseurText,
                    area: area,
                    prixUnitaire: prixMetreCarre,
                    cost: itemCost,
                });

                recapitulatif.etape4.totalArea += area;
                recapitulatif.etape4.totalCost += itemCost;
                total += itemCost;
            }
        });

    // ÉTAPE 5 - Bordures
    const bordureSelect = document.querySelector('select[name="bordure_id"]');
    if (
        bordureSelect?.value &&
        bordureSelect.value !== "" &&
        bordureSelect.value !== "0"
    ) {
        const selectedOption =
            bordureSelect.options[bordureSelect.selectedIndex];
        let bordurePrix =
            parseFloat(selectedOption.getAttribute("data-prix")) || 0;

        if (bordurePrix === 0 && selectedOption.text) {
            const prixMatch = selectedOption.text.match(/(\d+(?:\.\d+)?)\s*€/);
            if (prixMatch) {
                bordurePrix = parseFloat(prixMatch[1]);
            }
        }

        recapitulatif.etape5.data = {
            bordure: selectedOption?.text || "",
            prix: bordurePrix,
        };

        total += bordurePrix;
    }

    // ÉTAPE 6 - Options de découpe
    document.querySelectorAll(".optionsdecoupe-quantity").forEach((select) => {
        const quantity = parseInt(select.value) || 0;
        if (quantity > 0) {
            const prix = parseFloat(select.dataset.prix) || 0;
            const optionText =
                select.closest(".col-md-3")?.querySelector(".form-label")
                    ?.textContent ||
                select.closest(".col-md-3")?.querySelector("label")
                    ?.textContent ||
                "";
            const itemCost = quantity * prix;

            recapitulatif.etape6.data.push({
                option: optionText,
                quantity: quantity,
                prixUnitaire: prix,
                cost: itemCost,
            });

            recapitulatif.etape6.totalCost += itemCost;
            total += itemCost;
        }
    });

    // ÉTAPE 7 - Services
    document
        .querySelectorAll(
            'input[name*="services"]:checked, .service-checkbox:checked',
        )
        .forEach((checkbox) => {
            const prix = parseFloat(checkbox.dataset.prix) || 0;
            const serviceText =
                checkbox.closest(".form-check")?.querySelector("label")
                    ?.textContent ||
                checkbox.closest(".col-md-6")?.querySelector("label")
                    ?.textContent ||
                "";

            recapitulatif.etape7.data.push({
                service: serviceText,
                cost: prix,
            });

            recapitulatif.etape7.totalCost += prix;
            total += prix;
        });

    // ÉTAPE 8 - Accessoires
    document
        .querySelectorAll('input[name="accessoires[]"]:checked')
        .forEach((checkbox) => {
            const prix = parseFloat(checkbox.dataset.prix) || 0;
            const accessoireText =
                checkbox.closest(".form-check")?.querySelector("label")
                    ?.textContent || "";

            recapitulatif.etape8.data.push({
                accessoire: accessoireText,
                cost: prix,
            });

            recapitulatif.etape8.totalCost += prix;
            total += prix;
        });

    // Mettre à jour la section .pdt-name
    if (recapitulatif.etape3 && recapitulatif.etape3.data.length > 0) {
        updatePdtNameSection(recapitulatif.etape3);
    }

    // Mettre à jour les totaux dans le sidebar
    updateSidebarTotals(total);

    return { recapitulatif, total };
}

// Fonction pour mettre à jour la section .pdt-name
function updatePdtNameSection(etape3Data) {
    const pdtNameElements = document.querySelectorAll(".pdt-name");
    if (pdtNameElements.length === 0) return;

    // Vider d'abord tous les éléments .pdt-name
    pdtNameElements.forEach((element) => {
        element.innerHTML = "";
    });

    // Ajouter les informations des plans de travail
    if (etape3Data && etape3Data.data.length > 0) {
        etape3Data.data.forEach((plan, index) => {
            if (
                plan.longueur > 0 &&
                plan.largeur > 0 &&
                plan.epaisseur &&
                plan.epaisseur !== ""
            ) {
                if (pdtNameElements[index]) {
                    pdtNameElements[index].innerHTML =
                        `Plan ${plan.index}: ${plan.longueur}mm × ${plan.largeur}mm - ${plan.epaisseur} : ${plan.cost.toLocaleString("fr-FR", { style: "currency", currency: "EUR" })}`;
                } else {
                    const lastElement =
                        pdtNameElements[pdtNameElements.length - 1];
                    if (lastElement) {
                        lastElement.innerHTML += `<br>Plan ${plan.index}: ${plan.longueur}mm × ${plan.largeur}mm - ${plan.epaisseur} : ${plan.cost.toLocaleString("fr-FR", { style: "currency", currency: "EUR" })}`;
                    }
                }
            } else {
                if (pdtNameElements[index]) {
                    pdtNameElements[index].innerHTML = "";
                }
            }
        });
    }
}

// Fonction pour mettre à jour les totaux dans le sidebar
function updateSidebarTotals(total) {
    const totalAmountElement = document.getElementById("total-amount");
    const totalTtcElement = document.getElementById("total-ttc");

    if (totalAmountElement) {
        totalAmountElement.textContent = total.toLocaleString("fr-FR", {
            style: "currency",
            currency: "EUR",
        });
    }

    if (totalTtcElement) {
        // Calculer la TVA (20% par défaut, 10% si case cochée)
        const tvaCheckbox = document.querySelector('input[name="tva"]');
        const tvaRate = tvaCheckbox && tvaCheckbox.checked ? 0.1 : 0.2;
        const totalTtc = total * (1 + tvaRate);
        totalTtcElement.textContent = totalTtc.toLocaleString("fr-FR", {
            style: "currency",
            currency: "EUR",
        });
    }
}

// Fonction pour attacher les événements de calcul
function attachCalculationListeners() {
    // Attacher aux inputs de longueur et largeur
    document
        .querySelectorAll('input[name*="longueur"], input[name*="largeur"]')
        .forEach((input) => {
            ["input", "change", "keyup"].forEach((event) => {
                input.removeEventListener(event, calculateTotal);
                input.addEventListener(event, calculateTotal);
            });
        });

    // Attacher aux selects d'épaisseur
    document.querySelectorAll("select.epaisseur").forEach((select) => {
        ["change", "input"].forEach((event) => {
            select.removeEventListener(event, calculateTotal);
            select.addEventListener(event, calculateTotal);
        });
    });

    // Attacher au select de bordure
    const bordureSelect = document.querySelector('select[name="bordure_id"]');
    if (bordureSelect) {
        bordureSelect.removeEventListener("change", calculateTotal);
        bordureSelect.addEventListener("change", calculateTotal);
    }

    // Attacher aux options de découpe
    document.querySelectorAll(".optionsdecoupe-quantity").forEach((select) => {
        select.removeEventListener("change", calculateTotal);
        select.addEventListener("change", calculateTotal);
    });

    // Attacher aux services
    document
        .querySelectorAll('input[name*="services"], .service-checkbox')
        .forEach((checkbox) => {
            checkbox.removeEventListener("change", calculateTotal);
            checkbox.addEventListener("change", calculateTotal);
        });

    // Attacher aux accessoires
    document
        .querySelectorAll('input[name="accessoires[]"]')
        .forEach((checkbox) => {
            checkbox.removeEventListener("change", calculateTotal);
            checkbox.addEventListener("change", calculateTotal);
        });
}

// Gestion des accordéons avec validation
function initWizardAccordion() {
    const accordionElement = document.querySelector("#stepsdevis");
    if (!accordionElement) return;

    // Gérer les clics sur les boutons "Suivant"
    document.querySelectorAll(".nextstep").forEach((button) => {
        button.addEventListener("click", function (e) {
            e.preventDefault();
            let isValid = true;

            // Nettoyage des anciens messages
            const accordionItem = this.closest(".accordion-item");
            accordionItem
                .querySelectorAll("label.error-message")
                .forEach((el) => el.remove());
            accordionItem
                .querySelectorAll("select,input,textarea")
                .forEach((el) => el.classList.remove("is-invalid"));

            // Validation des champs requis
            accordionItem
                .querySelectorAll(
                    "select[required], input[required], textarea[required]",
                )
                .forEach((field) => {
                    const value = field.value;
                    if (!value || value === "") {
                        isValid = false;
                        field.classList.add("is-invalid");
                        const label =
                            document.querySelector(
                                `label[for="${field.id}"]`,
                            ) || field.previousElementSibling;
                        if (label) {
                            label.insertAdjacentHTML(
                                "afterend",
                                '<label class="error-message text-danger">Ce champ est requis.</label>',
                            );
                        }
                    }

                    // Validation email
                    if (field.type === "email" && value) {
                        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                        if (!emailRegex.test(value)) {
                            isValid = false;
                            field.classList.add("is-invalid");
                            const label =
                                document.querySelector(
                                    `label[for="${field.id}"]`,
                                ) || field.previousElementSibling;
                            if (label) {
                                label.insertAdjacentHTML(
                                    "afterend",
                                    '<label class="error-message text-danger">Adresse email invalide.</label>',
                                );
                            }
                        }
                    }
                });

            // Ouvrir l'accordéon suivant si validation OK
            if (isValid) {
                const nextAccordion = accordionItem.nextElementSibling;
                if (
                    nextAccordion &&
                    nextAccordion.classList.contains("accordion-item")
                ) {
                    const nextCollapse = nextAccordion.querySelector(
                        ".accordion-collapse",
                    );
                    if (nextCollapse) {
                        const bsCollapse = new bootstrap.Collapse(
                            nextCollapse,
                            { show: true },
                        );

                        // Scroll vers le nouvel accordéon
                        setTimeout(() => {
                            nextCollapse.scrollIntoView({
                                behavior: "smooth",
                                block: "start",
                            });
                        }, 300);
                    }
                }
            }
        });
    });
}

// Gestion des repeaters
function initRepeaters() {
    $("#kt_docs_repeater_basic, #kt_docs_repeater_credence").repeater({
        initEmpty: false,
        defaultValues: {
            "text-input": "foo",
        },
        show: function () {
            $(this).slideDown();

            // Traiter les selects normaux
            $(this)
                .find('select[data-control="select2"]:not(.epaisseur)')
                .each(function () {
                    if ($(this).hasClass("select2-hidden-accessible")) {
                        $(this).select2("destroy");
                    }
                    const hideSearch =
                        $(this).data("hide-search") === true ||
                        $(this).data("hide-search") === "true";
                    $(this).select2({
                        minimumResultsForSearch: hideSearch ? Infinity : 0,
                    });
                    $(this)
                        .val($(this).find("option:first").val())
                        .trigger("change");
                });

            // Peupler les nouveaux selects d'épaisseur
            $(this)
                .find("select.epaisseur")
                .each(function () {
                    populateEpaisseurSelect(this);
                    // Attacher immédiatement les événements pour ce select
                    ["change", "input"].forEach((event) => {
                        this.removeEventListener(event, calculateTotal);
                        this.addEventListener(event, calculateTotal);
                    });
                });

            // Attacher les événements de calcul pour tous les autres éléments
            attachCalculationListeners();
        },
        hide: function (deleteElement) {
            const repeaterContainer = $(this).closest("[data-repeater-list]");
            const itemCount = repeaterContainer.find(
                "[data-repeater-item]",
            ).length;

            if (itemCount <= 1) {
                alert(
                    "Impossible de supprimer le dernier élément. Au moins un élément doit être présent.",
                );
                return;
            }

            if (confirm("Êtes-vous sûr de vouloir supprimer cet élément ?")) {
                $(this).slideUp(deleteElement);
                setTimeout(() => calculateTotal(), 100);
            }
        },
    });
}

// Gestion du filtrage des coloris/finitions/épaisseurs
function initMaterialFiltering() {
    if (typeof axios === "undefined") return;

    // Configuration axios
    axios.defaults.headers.common["X-CSRF-TOKEN"] =
        document
            .querySelector('meta[name="csrf-token"]')
            ?.getAttribute("content") || "";

    const matiereSelect = document.querySelector('select[name="matiere_id"]');
    const coloriSelect = document.querySelector('select[name="colori_id"]');
    const finitionSelect = document.querySelector('select[name="finition_id"]');

    if (!matiereSelect || !coloriSelect) return;

    // Changement de matière
    $("#matiere_id").on("change", function () {
        const matiereId = this.value;
        loadColorisForMatiere(matiereId);
    });

    // Changement de coloris
    $("#colori_id").on("change", function () {
        const coloriId = this.value;
        loadFinitionsForColoris(coloriId);
        resetRepeaterItems();
    });

    // Changement de finition
    $(document).on("change", 'select[name="finition_id"]', function () {
        const finitionId = this.value;
        const coloriId = $("#colori_id").val();
        if (coloriId && finitionId) {
            loadEpaisseursByColoriFinition(coloriId, finitionId);
        }
        resetRepeaterItems();
    });

    // Fonctions de chargement
    function loadColorisForMatiere(matiereId, selectedColoriId = null) {
        coloriSelect.innerHTML =
            '<option value="">Sélectionner un coloris</option>';

        if (matiereId) {
            const url = `${window.location.origin}/admin/coloris/by-matiere/${matiereId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const coloris = response.data;
                    if (Array.isArray(coloris) && coloris.length > 0) {
                        coloris.forEach((colori) => {
                            const option = document.createElement("option");
                            option.value = colori.id;
                            option.textContent = colori.titre;
                            if (
                                selectedColoriId &&
                                selectedColoriId == colori.id
                            ) {
                                option.selected = true;
                            }
                            coloriSelect.appendChild(option);
                        });
                    }
                    if (typeof $(coloriSelect).select2 === "function") {
                        $(coloriSelect).trigger("change");
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des coloris:",
                        error,
                    );
                });
        }
    }

    function loadFinitionsForColoris(coloriId, selectedFinitionId = null) {
        if (!finitionSelect) return;

        finitionSelect.innerHTML =
            '<option value="">Sélectionner une finition</option>';

        if (coloriId) {
            const url = `${window.location.origin}/admin/finitions/by-coloris/${coloriId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const finitions = response.data;
                    if (Array.isArray(finitions) && finitions.length > 0) {
                        finitions.forEach((finition) => {
                            const option = document.createElement("option");
                            option.value = finition.id;
                            option.textContent = finition.titre;
                            if (
                                selectedFinitionId &&
                                selectedFinitionId == finition.id
                            ) {
                                option.selected = true;
                            }
                            finitionSelect.appendChild(option);
                        });
                    }
                    if (typeof $(finitionSelect).select2 === "function") {
                        $(finitionSelect).trigger("change");
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des finitions:",
                        error,
                    );
                });
        }
    }

    function loadEpaisseursByColoriFinition(coloriId, finitionId) {
        const epaisseurSelects = document.querySelectorAll("select.epaisseur");
        epaisseurSelects.forEach((select) => {
            select.innerHTML =
                '<option value="">Choisissez une épaisseur</option>';
        });

        if (coloriId && finitionId) {
            const url = `${window.location.origin}/admin/epaisseurs/by-coloris-finition/${coloriId}/${finitionId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const epaisseurs = response.data;
                    window.currentEpaisseurs = epaisseurs;

                    if (Array.isArray(epaisseurs) && epaisseurs.length > 0) {
                        epaisseurSelects.forEach((select) => {
                            populateEpaisseurSelect(select);
                        });
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des épaisseurs:",
                        error,
                    );
                    window.currentEpaisseurs = [];
                });
        }
    }

    function resetRepeaterItems() {
        ["kt_docs_repeater_basic", "kt_docs_repeater_credence"].forEach(
            (repeaterId) => {
                const repeaterList = document.querySelector(
                    `[data-repeater-list="${repeaterId}"]`,
                );
                if (repeaterList) {
                    const firstItem = repeaterList.querySelector(
                        "[data-repeater-item]",
                    );
                    const otherItems = repeaterList.querySelectorAll(
                        "[data-repeater-item]:not(:first-child)",
                    );
                    otherItems.forEach((item) => item.remove());

                    if (firstItem) {
                        firstItem
                            .querySelectorAll("input, select")
                            .forEach((field) => {
                                field.value = "";
                                if (field.tagName === "SELECT") {
                                    $(field).trigger("change");
                                }
                            });
                    }
                }
            },
        );
    }

    function updatePreview() {
        const matiereText =
            matiereSelect.options[matiereSelect.selectedIndex]?.text;
        const coloriText =
            coloriSelect.options[coloriSelect.selectedIndex]?.text;
        const finitionText =
            finitionSelect?.options[finitionSelect.selectedIndex]?.text;

        const materialNameElement = document.getElementById("material-name");
        const coloriNameElement = document.getElementById("colori-name");
        const finishNameElement = document.getElementById("finish-name");

        if (
            materialNameElement &&
            matiereText &&
            matiereText !== "Sélectionner une matière"
        ) {
            materialNameElement.textContent = matiereText;
        }
        if (
            coloriNameElement &&
            coloriText &&
            coloriText !== "Sélectionner un coloris"
        ) {
            coloriNameElement.textContent = " - " + coloriText;
        }
        if (
            finishNameElement &&
            finitionText &&
            finitionText !== "Sélectionner une finition"
        ) {
            finishNameElement.textContent = finitionText;
        }
    }

    // Charger les données initiales si en mode édition
    const initialMatiereId = matiereSelect.value;
    if (initialMatiereId) {
        const selectedColoriId = coloriSelect.value;
        loadColorisForMatiere(initialMatiereId, selectedColoriId);

        if (selectedColoriId) {
            const selectedFinitionId = finitionSelect?.value;
            setTimeout(() => {
                loadFinitionsForColoris(selectedColoriId, selectedFinitionId);
            }, 500);
        }
    }
}

// Gestion du changement d'image de bordure
function initBordureImageChange() {
    const bordureSelect = document.getElementById("bordure_id");
    const bordureImage = document.getElementById("bordureimage");

    if (bordureSelect && bordureImage) {
        bordureSelect.addEventListener("change", function () {
            const selectedOption = this.options[this.selectedIndex];
            let photoUrl = selectedOption.getAttribute("data-photo");

            if (
                photoUrl &&
                photoUrl !== "" &&
                photoUrl !== "null" &&
                photoUrl !== "undefined"
            ) {
                bordureImage.src = photoUrl;
                bordureImage.onload = () =>
                    console.log("Image chargée avec succès");
                bordureImage.onerror = () => {
                    const defaultImg =
                        "/assets/media/avatars/chant-droit-biseaute.jpg";
                    bordureImage.src = defaultImg;
                };
            } else {
                const defaultImg =
                    "/assets/media/avatars/chant-droit-biseaute.jpg";
                bordureImage.src = defaultImg;
            }
        });

        // Déclencher le changement au chargement si une bordure est déjà sélectionnée
        if (bordureSelect.value) {
            setTimeout(() => {
                bordureSelect.dispatchEvent(new Event("change"));
            }, 100);
        }
    }
}

// Initialisation globale
document.addEventListener("DOMContentLoaded", function () {
    // Initialiser tous les composants
    initWizardAccordion();
    initMaterialFiltering();
    initBordureImageChange();
    attachCalculationListeners();

    // Calculer le total initial
    setTimeout(() => {
        calculateTotal();
    }, 500);
});

// Initialisation avec jQuery si disponible
if (typeof $ !== "undefined") {
    $(document).ready(function () {
        initRepeaters();
        attachCalculationListeners();
    });
}
