// Vérifier si on est en mode édition
const isEditMode =
    $('input[name="_method"]').length > 0 &&
    $('input[name="_method"]').val() === "PUT";

if (isEditMode) {
    // En mode édition : activer tous les accordéons et les ouvrir
    $(".accordion-button").attr("aria-disabled", false).attr("disabled", false);
    $(".accordion-collapse").addClass("show");
} else {
    // En mode création : désactiver tous sauf le premier
    $(".accordion-button").attr("aria-disabled", true).attr("disabled", true);
    $(".accordion-button")
        .first()
        .attr("disabled", false)
        .attr("aria-disabled", false);
}

// STEP 1 -------------------
$(".nextstep").on("click", function (e) {
    e.preventDefault();
    let isValid = true;

    $(".accordion-item").find("label.error-message").remove();
    $(".accordion-item")
        .find("select,input,textarea")
        .removeClass("is-invalid");

    const accordionItem = $(this).closest(".accordion-item");

    accordionItem
        .find("select[required], input[required], textarea[required]")
        .each(function () {
            const $select = $(this);
            const selected = $select.val();
            const selectId = $select.attr("id");
            const type = $select.attr("type");
            const $label = $('label[for="' + selectId + '"]');

            if (!selected || selected === "") {
                isValid = false;
                $select.addClass("is-invalid");
                $label.after(
                    '<label class="error-message text-danger">Ce champ est requis.</label>',
                );
            }

            // Si email invalide
            if (type === "email") {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailRegex.test(selected)) {
                    isValid = false;
                    $select.addClass("is-invalid");
                    $label.after(
                        '<label class="error-message"> Adresse email invalide.</label>',
                    );
                }
            }

            // Validation pour le champ addperc (doit être un entier entre 0 et 100)
            if (selectId === "addperc" && type === "number") {
                const value = parseInt(selected);
                if (
                    selected !== "" &&
                    (isNaN(value) ||
                        value < 0 ||
                        value > 100 ||
                        selected.includes(".") ||
                        selected.includes(","))
                ) {
                    isValid = false;
                    $select.addClass("is-invalid");
                    $label.after(
                        '<label class="error-message text-danger">La valeur doit être un nombre entier entre 0 et 100 (pas de virgules).</label>',
                    );
                }
            }
        });

    // Si tout est OK → ouvrir le 2e accordion
    if (isValid) {
        let nextAccordion = accordionItem.next(".accordion-item");
        let nextCollapse = nextAccordion.find(".accordion-collapse");
        nextCollapse.collapse("show");

        const btn2 = $(".accordion-button.accordion2");
        btn2.attr("aria-disabled", false).attr("disabled", false);

        $("html, body").animate(
            {
                scrollTop: nextCollapse.offset().top - 200, // -100 pour un petit décalage visuel
            },
            500,
        );
    }
});

// STEP 2 -------------------
const matiereSelect = document.querySelector("#matiere_id");
const coloriSelect = document.querySelector("#colori_id");
const finitionSelect = document.querySelector('select[name="finition_id"]');

function initMaterialFiltering() {
    if (typeof axios === "undefined") return;

    // Configuration axios
    axios.defaults.headers.common["X-CSRF-TOKEN"] =
        document
            .querySelector('meta[name="csrf-token"]')
            ?.getAttribute("content") || "";

    if (!matiereSelect || !coloriSelect) return;

    let isFormSubmitting = false;
    const isEditMode =
        $('input[name="_method"]').length > 0 &&
        $('input[name="_method"]').val() === "PUT";

    // Prevent AJAX calls during form submission
    $("form").on("submit", function () {
        isFormSubmitting = true;
    });

    // Matiere change event
    $("#matiere_id").change(function () {
        if (isFormSubmitting || isEditMode) return;

        const matiereId = $(this).val();
        const coloriSelect = $("#colori_id");

        // Clear coloris and dependent fields
        coloriSelect.innerHTML =
            '<option value="">Sélectionner un coloris</option>';
        $("#finition_id")
            .empty()
            .append('<option value="">Sélectionner une finition</option>');
        resetRepeaterItems();

        if (matiereId) {
            loadColorisForMatiere(matiereId);
        }
    });

    // Coloris change event
    $("#colori_id").change(function () {
        if (isFormSubmitting || isEditMode) return;

        const coloriId = $(this).val();
        const finitionSelect = $("#finition_id");

        // Clear finitions and dependent fields
        finitionSelect.innerHTML =
            '<option value="">Sélectionner une finition</option>';
        resetRepeaterItems();

        if (coloriId) {
            loadFinitionsForColoris(coloriId);
            getPhotoColoris();
        }
    });

    // Finition change event
    $(document).on("change", 'select[name="finition_id"]', function () {
        if (isFormSubmitting) return;

        const finitionId = this.value;
        const coloriId = $("#colori_id").val();
        if (coloriId && finitionId) {
            loadEpaisseursByColoriFinition(coloriId, finitionId);
        }
        if (!isEditMode) {
            resetRepeaterItems();
        }
    });

    // Fonctions de chargement
    function loadColorisForMatiere(matiereId, selectedColoriId = null) {
        coloriSelect.innerHTML =
            '<option value="">Sélectionner un coloris</option>';

        if (matiereId) {
            const url = `${window.location.origin}/admin/coloris/by-matiere/${matiereId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const coloris = response.data;
                    if (Array.isArray(coloris) && coloris.length > 0) {
                        coloris.forEach((colori) => {
                            const option = document.createElement("option");
                            option.value = colori.id;
                            option.textContent = colori.titre;
                            if (
                                selectedColoriId &&
                                selectedColoriId == colori.id
                            ) {
                                option.selected = true;
                            }
                            coloriSelect.appendChild(option);
                        });
                    }
                    if (typeof $(coloriSelect).select2 === "function") {
                        $(coloriSelect).trigger("change");
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des coloris:",
                        error,
                    );
                });
        }
    }

    function loadFinitionsForColoris(coloriId, selectedFinitionId = null) {
        if (!finitionSelect) return;

        finitionSelect.innerHTML =
            '<option value="">Sélectionner une finition</option>';

        if (coloriId) {
            const url = `${window.location.origin}/admin/finitions/by-coloris/${coloriId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const finitions = response.data;
                    if (Array.isArray(finitions) && finitions.length > 0) {
                        finitions.forEach((finition) => {
                            const option = document.createElement("option");
                            option.value = finition.id;
                            option.textContent = finition.titre;
                            if (
                                selectedFinitionId &&
                                selectedFinitionId == finition.id
                            ) {
                                option.selected = true;
                            }
                            finitionSelect.appendChild(option);
                        });
                    }
                    if (typeof $(finitionSelect).select2 === "function") {
                        $(finitionSelect).trigger("change");
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des finitions:",
                        error,
                    );
                });
        }
    }

    function loadEpaisseursByColoriFinition(coloriId, finitionId) {
        const epaisseurSelects = document.querySelectorAll("select.epaisseur");

        // En mode édition, sauvegarder les data-selected initiaux des selects
        const initialSelectedValues = [];
        epaisseurSelects.forEach((select, index) => {
            if (isEditMode) {
                // Prendre la valeur du data-selected s'il existe, sinon la valeur actuelle
                const dataSelected = select.getAttribute("data-selected");
                const currentValue = select.value;
                initialSelectedValues[index] =
                    dataSelected || currentValue || null;
                console.log(
                    `💾 Valeur initiale pour select ${index}: ${initialSelectedValues[index]}`,
                );
            }
        });

        // Vider tous les selects
        epaisseurSelects.forEach((select) => {
            select.innerHTML =
                '<option value="">Choisissez une épaisseur</option>';
        });

        if (coloriId && finitionId) {
            const url = `${window.location.origin}/admin/epaisseurs/by-coloris-finition/${coloriId}/${finitionId}`;
            axios
                .get(url.replace("http://", "https://"))
                .then((response) => {
                    const epaisseurs = response.data;
                    window.currentEpaisseurs = epaisseurs;
                    console.log(
                        `📦 ${epaisseurs.length} épaisseurs reçues pour coloris ${coloriId} et finition ${finitionId}`,
                    );
                    console.log("🔍 Structure des épaisseurs:", epaisseurs);

                    if (Array.isArray(epaisseurs) && epaisseurs.length > 0) {
                        epaisseurSelects.forEach((select, index) => {
                            // Peupler le select avec les nouvelles épaisseurs
                            const addperc = getAddpercValue();
                            epaisseurs.forEach((epaisseur) => {
                                const option = document.createElement("option");
                                const optionValue =
                                    epaisseur.prix_colori_id || epaisseur.id;
                                option.value = optionValue;

                                // Appliquer le pourcentage addperc au prix affiché
                                const originalPrice = parseFloat(
                                    epaisseur.prix_raw || epaisseur.prix,
                                );
                                const displayPrice = applyAddpercToPrice(
                                    originalPrice,
                                    addperc,
                                );

                                option.textContent = `${epaisseur.titre} - ${displayPrice.toFixed(2)}€/m²`;
                                option.setAttribute(
                                    "data-prix",
                                    epaisseur.prix_raw || epaisseur.prix,
                                );
                                option.setAttribute(
                                    "data-prix-colori-id",
                                    epaisseur.prix_colori_id,
                                );

                                select.appendChild(option);
                            });

                            // En mode édition, restaurer la valeur initiale
                            if (isEditMode && initialSelectedValues[index]) {
                                const valueToRestore =
                                    initialSelectedValues[index].toString();
                                console.log(
                                    `🔍 Tentative de restauration de la valeur ${valueToRestore} pour le select ${index}`,
                                );

                                // Chercher et sélectionner l'option correspondante
                                let optionFound = false;

                                // Essayer d'abord avec la valeur exacte
                                Array.from(select.options).forEach((option) => {
                                    if (option.value === valueToRestore) {
                                        option.selected = true;
                                        optionFound = true;
                                        console.log(
                                            `✅ Valeur restaurée avec succès (exact match): ${option.textContent} (${option.value})`,
                                        );
                                    }
                                });

                                // Si pas trouvé, essayer de chercher par ID d'épaisseur
                                if (!optionFound) {
                                    const matchingEpaisseur = epaisseurs.find(
                                        (ep) =>
                                            ep.id.toString() ===
                                                valueToRestore ||
                                            ep.prix_colori_id.toString() ===
                                                valueToRestore,
                                    );

                                    if (matchingEpaisseur) {
                                        Array.from(select.options).forEach(
                                            (option) => {
                                                if (
                                                    option.value ===
                                                    matchingEpaisseur.prix_colori_id.toString()
                                                ) {
                                                    option.selected = true;
                                                    optionFound = true;
                                                    console.log(
                                                        `✅ Valeur restaurée via correspondance: ${option.textContent} (${option.value})`,
                                                    );
                                                }
                                            },
                                        );
                                    }
                                }

                                if (!optionFound) {
                                    console.log(
                                        `❌ Valeur ${valueToRestore} non trouvée dans les nouvelles options`,
                                    );
                                    console.log(
                                        "📋 Options disponibles:",
                                        Array.from(select.options).map(
                                            (opt) => opt.value,
                                        ),
                                    );
                                }
                            }

                            // Réinitialiser Select2 et déclencher les événements
                            if (
                                $(select).hasClass("select2-hidden-accessible")
                            ) {
                                $(select).select2("destroy");
                            }
                            $(select).select2({
                                minimumResultsForSearch: Infinity,
                            });

                            // Attacher l'événement change pour déclencher les calculs
                            $(select).on("change", function () {
                                calculateTotal();
                                updatePdtRecap();
                                updateCredenceRecap();
                            });

                            // Déclencher l'événement change pour mettre à jour les calculs
                            $(select).trigger("change");
                        });
                    }
                    updatePreview();
                })
                .catch((error) => {
                    console.error(
                        "Erreur lors du chargement des épaisseurs:",
                        error,
                    );
                    window.currentEpaisseurs = [];
                });
        }
    }

    // Charger les données initiales
    const initialMatiereId = matiereSelect.value;
    if (initialMatiereId) {
        const selectedColoriId = coloriSelect.value;

        if (!isEditMode) {
            loadColorisForMatiere(initialMatiereId, selectedColoriId);
        }

        if (selectedColoriId) {
            const selectedFinitionId = finitionSelect?.value;

            if (!isEditMode) {
                setTimeout(() => {
                    loadFinitionsForColoris(
                        selectedColoriId,
                        selectedFinitionId,
                    );
                }, 500);
            }

            // Charger les épaisseurs si finition est sélectionnée
            if (selectedFinitionId) {
                console.log(
                    `Chargement des épaisseurs - Mode édition: ${isEditMode}`,
                );

                setTimeout(
                    () => {
                        loadEpaisseursByColoriFinition(
                            selectedColoriId,
                            selectedFinitionId,
                        );
                    },
                    isEditMode ? 1000 : 500,
                );
            }
        }
    }

    // En mode édition, s'assurer que le calcul se fait après le chargement des épaisseurs
    if (isEditMode) {
        console.log(
            "🔄 Mode édition détecté - Programmation du recalcul différé...",
        );

        // Fonction de recalcul complet
        function forceRecalculate() {
            console.log("🔄 Recalcul forcé en mode édition...");
            calculateTotal();
            updatePdtRecap();
            updateCredenceRecap();
            updateOptionsRecap();
            updateServicesRecap();
            updateBordureRecap();
            updateTvaRecap();
        }

        // Recalculs multiples pour s'assurer que tout est bien chargé
        setTimeout(forceRecalculate, 1000);
        setTimeout(forceRecalculate, 2500);
        setTimeout(forceRecalculate, 4000);

        // Observer les changements dans les selects d'épaisseur pour déclencher le recalcul
        const observer = new MutationObserver(function (mutations) {
            mutations.forEach(function (mutation) {
                if (
                    mutation.type === "childList" &&
                    mutation.target.classList.contains("epaisseur")
                ) {
                    console.log(
                        "🔄 Changement détecté dans un select épaisseur - Recalcul...",
                    );
                    setTimeout(forceRecalculate, 500);
                }
            });
        });

        // Observer tous les selects d'épaisseur
        document.querySelectorAll("select.epaisseur").forEach((select) => {
            observer.observe(select, { childList: true, subtree: true });
        });
    }
}

//recap: Matiere-Coloris-Finition
function updatePreview() {
    const matiereText =
        matiereSelect.options[matiereSelect.selectedIndex]?.text;
    const coloriText = coloriSelect.options[coloriSelect.selectedIndex]?.text;
    const finitionText =
        finitionSelect?.options[finitionSelect.selectedIndex]?.text;

    const materialNameElement = document.getElementById("material-name");
    const coloriNameElement = document.getElementById("colori-name");
    const finishNameElement = document.getElementById("finish-name");

    if (
        materialNameElement &&
        matiereText &&
        matiereText !== "Sélectionner une matière"
    ) {
        materialNameElement.textContent = matiereText;
    }
    if (
        coloriNameElement &&
        coloriText &&
        coloriText !== "Sélectionner un coloris"
    ) {
        coloriNameElement.textContent = " - " + coloriText;
    }
    if (
        finishNameElement &&
        finitionText &&
        finitionText !== "Sélectionner une finition"
    ) {
        finishNameElement.textContent = finitionText;
    }

    // Mettre à jour l'image du coloris
    $("#colori_id").on("change", function () {
        getPhotoColoris();
    });
}

function getPhotoColoris() {
    const materialPreviewElement = document.getElementById("material-preview");
    const coloriId = coloriSelect.value;
    if (coloriId && materialPreviewElement) {
        // Charger la photo du coloris sélectionné

        const url = `${window.location.origin}/admin/coloris/${coloriId}/photo`;
        axios
            .get(url.replace("http://", "https://"))
            .then((response) => {
                console.log(
                    "Chargement de la photo du coloris:",
                    response.data,
                );
                if (response.data && response.data.photo) {
                    materialPreviewElement.src = `/storage/${response.data.photo}`;
                    materialPreviewElement.onerror = () => {
                        // Image par défaut si erreur de chargement
                        materialPreviewElement.src =
                            "/assets/media/avatars/granit-noir-zimbabwe-extra.jpg";
                    };
                } else {
                    // Image par défaut si pas de photo
                    materialPreviewElement.src =
                        "/assets/media/avatars/granit-noir-zimbabwe-extra.jpg";
                }
            })
            .catch((error) => {
                console.error(
                    "Erreur lors du chargement de la photo du coloris:",
                    error,
                );
                // Image par défaut en cas d'erreur
                materialPreviewElement.src =
                    "/assets/media/avatars/granit-noir-zimbabwe-extra.jpg";
            });
    } else if (materialPreviewElement) {
        // Image par défaut si aucun coloris sélectionné
        materialPreviewElement.src =
            "/assets/media/avatars/granit-noir-zimbabwe-extra.jpg";
    }
}

// Fonction pour mettre à jour le récapitulatif des plans de travail
function updatePdtRecap() {
    const pdtContainer = document.getElementById("pdt-id");
    if (!pdtContainer) return;

    // Vider le conteneur
    pdtContainer.innerHTML = "";

    let planIndex = 1;
    const addperc = getAddpercValue();

    // Parcourir tous les éléments repeater des plans de travail
    document
        .querySelectorAll("#kt_docs_repeater_plantravail [data-repeater-item]")
        .forEach((item) => {
            const longueurInput = item.querySelector('input[name*="longueur"]');
            const largeurInput = item.querySelector('input[name*="largeur"]');
            const epaisseurSelect = item.querySelector(
                'select[name*="epaisseur"]',
            );

            if (!longueurInput || !largeurInput || !epaisseurSelect) return;

            const longueur = parseFloat(longueurInput.value) || 0;
            const largeur = parseFloat(largeurInput.value) || 0;
            const epaisseurId = epaisseurSelect.value;

            // Vérifier si tous les champs sont remplis et valides
            if (longueur > 0 && largeur > 0 && epaisseurId) {
                const selectedOption =
                    epaisseurSelect.options[epaisseurSelect.selectedIndex];
                const epaisseurText = selectedOption
                    ? selectedOption.textContent
                    : "";
                const prixOriginal =
                    parseFloat(selectedOption?.getAttribute("data-prix")) || 0;

                // Appliquer le pourcentage addperc
                const prixAvecPourcentage = applyAddpercToPrice(
                    prixOriginal,
                    addperc,
                );

                // Calculer la surface en m²
                const surface = (longueur * largeur) / 1000000;
                const coutTotal = surface * prixAvecPourcentage;

                // Créer l'élément récapitulatif
                const recapDiv = document.createElement("div");
                recapDiv.className = "pdt-name mb-2";
                recapDiv.innerHTML = `
                <div class="fw-bold mt-5">Plan de travail ${planIndex}</div>
                <div class="small text-muted">${longueur}mm × ${largeur}mm - Ep: ${epaisseurText}</div>
                <div class="text-success fw-bold">${coutTotal.toFixed(2)}€</div>
            `;

                pdtContainer.appendChild(recapDiv);
                planIndex++;
            }
        });

    // Si aucun plan valide, afficher un message par défaut
    if (pdtContainer.innerHTML === "") {
        pdtContainer.innerHTML = '<div class="pdt-name text-muted"></div>';
    }
}

//reinitialiser les valeurs des epaisseurs
function resetRepeaterItems() {
    ["kt_docs_repeater_plantravail", "kt_docs_repeater_credence"].forEach(
        (repeaterId) => {
            const repeaterList = document.querySelector(
                `[data-repeater-list="${repeaterId}"]`,
            );
            if (repeaterList) {
                const firstItem = repeaterList.querySelector(
                    "[data-repeater-item]",
                );
                const otherItems = repeaterList.querySelectorAll(
                    "[data-repeater-item]:not(:first-child)",
                );
                otherItems.forEach((item) => item.remove());

                if (firstItem) {
                    firstItem
                        .querySelectorAll("input, select")
                        .forEach((field) => {
                            field.value = "";
                            if (field.tagName === "SELECT") {
                                $(field).trigger("change");
                            }
                        });
                }
            }
        },
    );
}

// STEP 3 -------------------
$("#kt_docs_repeater_plantravail, #kt_docs_repeater_credence").repeater({
    initEmpty: false,

    defaultValues: {
        "text-input": "foo",
    },

    show: function () {
        $(this).slideDown();

        // Traiter d'abord les selects normaux
        $(this)
            .find('select[data-control="select2"]:not(.epaisseur)')
            .each(function () {
                if ($(this).hasClass("select2-hidden-accessible")) {
                    $(this).select2("destroy");
                }
                const hideSearch =
                    $(this).data("hide-search") === true ||
                    $(this).data("hide-search") === "true";

                $(this).select2({
                    minimumResultsForSearch: hideSearch ? Infinity : 0,
                });
                $(this)
                    .val($(this).find("option:first").val())
                    .trigger("change");
            });

        // Peupler les nouveaux selects d'épaisseur avec les données actuelles
        $(this)
            .find("select.epaisseur")
            .each(function () {
                if (
                    window.currentEpaisseurs &&
                    window.currentEpaisseurs.length > 0
                ) {
                    const addperc = getAddpercValue();
                    window.currentEpaisseurs.forEach((epaisseur) => {
                        const option = document.createElement("option");
                        const optionValue =
                            epaisseur.prix_colori_id || epaisseur.id;
                        option.value = optionValue;

                        // Appliquer le pourcentage addperc au prix affiché
                        const originalPrice = parseFloat(
                            epaisseur.prix_raw || epaisseur.prix,
                        );
                        const displayPrice = applyAddpercToPrice(
                            originalPrice,
                            addperc,
                        );

                        option.textContent = `${epaisseur.titre} - ${displayPrice.toFixed(2)}€/m²`;
                        option.setAttribute(
                            "data-prix",
                            epaisseur.prix_raw || epaisseur.prix,
                        );
                        option.setAttribute(
                            "data-prix-colori-id",
                            epaisseur.prix_colori_id,
                        );

                        this.appendChild(option);
                    });
                }

                // Réinitialiser Select2 si nécessaire
                if ($(this).hasClass("select2-hidden-accessible")) {
                    $(this).select2("destroy");
                }
                $(this).select2({
                    minimumResultsForSearch: Infinity,
                });

                // Attacher les événements de calcul
                ["change", "input"].forEach((event) => {
                    this.removeEventListener(event, calculateTotal);
                    this.addEventListener(event, calculateTotal);
                    this.removeEventListener(event, updatePdtRecap);
                    this.addEventListener(event, updatePdtRecap);
                    this.removeEventListener(event, updateCredenceRecap);
                    this.addEventListener(event, updateCredenceRecap);
                });
            });

        // Attacher les événements pour le récapitulatif aux nouveaux éléments
        $(this)
            .find(
                'input[name*="longueur"], input[name*="largeur"], select[name*="epaisseur"]',
            )
            .on("change input", function () {
                updatePdtRecap();
            });

        // Attacher les événements pour le récapitulatif des crédences aux nouveaux éléments
        $(this)
            .find(
                'input[name*="credence_longueur"], input[name*="credence_largeur"], select[name*="credence_epaisseur"]',
            )
            .on("change input", function () {
                updateCredenceRecap();
            });

        // Attacher les événements de calcul aux nouveaux éléments
        attachCalculationEvents();

        // Réinitialiser la validation en direct pour les nouveaux éléments
        //   setupLiveValidation();
    },

    hide: function (deleteElement) {
        // Compter le nombre d'éléments dans ce repeater
        const repeaterContainer = $(this).closest("[data-repeater-list]");
        const itemCount = repeaterContainer.find("[data-repeater-item]").length;

        if (itemCount <= 1) {
            $(this).find("input, select").val("");
            alert(
                "Impossible de supprimer le dernier élément. Au moins un élément doit être présent.",
            );
        } else {
            if (confirm("Êtes-vous sûr de vouloir supprimer cet élément ?")) {
                $(this).slideUp(deleteElement);
            }
        }
        // Recalculer après suppression et mettre à jour le récapitulatif
        setTimeout(function () {
            calculateTotal();
            updatePdtRecap();
            updateCredenceRecap();
        }, 450); // Augmenter le délai pour s'assurer que l'élément est bien supprimé
    },
});

// STEP 4 -------------------
function initBordureImageChange() {
    const bordureSelect = document.getElementById("bordure_id");
    const bordureImage = document.getElementById("bordureimage");

    if (bordureSelect && bordureImage) {
        bordureSelect.addEventListener("change", function () {
            const selectedOption = this.options[this.selectedIndex];
            let photoUrl = selectedOption.getAttribute("data-photo");

            if (
                photoUrl &&
                photoUrl !== "" &&
                photoUrl !== "null" &&
                photoUrl !== "undefined"
            ) {
                bordureImage.src = photoUrl;
                bordureImage.onload = () =>
                    console.log("Image chargée avec succès");
                bordureImage.onerror = () => {
                    const defaultImg =
                        "/assets/media/avatars/chant-droit-biseaute.jpg";
                    bordureImage.src = defaultImg;
                };
            } else {
                const defaultImg =
                    "/assets/media/avatars/chant-droit-biseaute.jpg";
                bordureImage.src = defaultImg;
            }

            // Mettre à jour le récapitulatif de la bordure
            updateBordureRecap();
        });

        // Déclencher le changement au chargement si une bordure est déjà sélectionnée
        if (bordureSelect.value) {
            setTimeout(() => {
                bordureSelect.dispatchEvent(new Event("change"));
            }, 100);
        }
    }
}

// Fonction pour mettre à jour le récapitulatif de la bordure
function updateBordureRecap() {
    const bordureSelect = document.getElementById("bordure_id");
    const bordureNameElement = document.getElementById("bordure-name");

    if (!bordureSelect || !bordureNameElement) return;

    const selectedValue = bordureSelect.value;
    const selectedOption = bordureSelect.options[bordureSelect.selectedIndex];

    const prixOriginal =
        parseFloat(selectedOption?.getAttribute("data-prix")) || 0;
    const addperc = getAddpercValue();
    const prixAvecPourcentage = applyAddpercToPrice(prixOriginal, addperc);

    // Vérifier si une bordure est sélectionnée et que la valeur n'est pas vide ou 0
    if (selectedValue && selectedValue !== "" && selectedValue !== "0") {
        const bordureName = selectedOption.textContent;

        bordureNameElement.innerHTML = `
            <div class="fw-bold mt-5">Bordure</div>
            <div class="small text-muted">${bordureName}</div>
            <div class="text-success fw-bold">${prixAvecPourcentage.toFixed(2)}€</div>
        `;
    } else {
        // Effacer le récapitulatif si aucune bordure n'est sélectionnée ou si la valeur est vide/0
        bordureNameElement.textContent = "";
    }
}

// Fonction pour mettre à jour le récapitulatif des crédences
function updateCredenceRecap() {
    const credenceContainer = document.getElementById("credences-id");
    if (!credenceContainer) return;

    // Vider le conteneur
    credenceContainer.innerHTML = "";

    let credenceIndex = 1;
    const addperc = getAddpercValue();

    // Parcourir tous les éléments repeater des crédences
    document
        .querySelectorAll("#kt_docs_repeater_credence [data-repeater-item]")
        .forEach((item) => {
            const longueurInput = item.querySelector(
                'input[name*="credence_longueur"]',
            );
            const largeurInput = item.querySelector(
                'input[name*="credence_largeur"]',
            );
            const epaisseurSelect = item.querySelector(
                'select[name*="credence_epaisseur"]',
            );

            if (!longueurInput || !largeurInput || !epaisseurSelect) return;

            const longueur = parseFloat(longueurInput.value) || 0;
            const largeur = parseFloat(largeurInput.value) || 0;
            const epaisseurId = epaisseurSelect.value;

            // Vérifier si tous les champs sont remplis et valides
            if (longueur > 0 && largeur > 0 && epaisseurId) {
                const selectedOption =
                    epaisseurSelect.options[epaisseurSelect.selectedIndex];
                const epaisseurText = selectedOption
                    ? selectedOption.textContent
                    : "";
                const prixOriginal =
                    parseFloat(selectedOption?.getAttribute("data-prix")) || 0;

                // Appliquer le pourcentage addperc
                const prixAvecPourcentage = applyAddpercToPrice(
                    prixOriginal,
                    addperc,
                );

                // Calculer la surface en m²
                const surface = (longueur * largeur) / 1000000;
                const coutTotal = surface * prixAvecPourcentage;

                // Créer l'élément récapitulatif
                const recapDiv = document.createElement("div");
                recapDiv.className = "credence-name mb-2";
                recapDiv.innerHTML = `
                <div class="fw-bold mt-5">Crédence ${credenceIndex}</div>
                <div class="small text-muted">${longueur}mm × ${largeur}mm - Ep: ${epaisseurText}</div>
                <div class="text-success fw-bold">${coutTotal.toFixed(2)}€</div>
            `;

                credenceContainer.appendChild(recapDiv);
                credenceIndex++;
            }
        });

    // Si aucune crédence valide, afficher un message par défaut
    if (credenceContainer.innerHTML === "") {
        credenceContainer.innerHTML =
            '<div class="credence-name text-muted"></div>';
    }
}

document.addEventListener("DOMContentLoaded", function () {
    // initWizardAccordion();
    initMaterialFiltering();
    initBordureImageChange();
    // attachCalculationListeners();

    // Initialiser le récapitulatif de bordure au chargement
    updateBordureRecap();

    // Attacher les événements pour le récapitulatif aux éléments existants
    document
        .querySelectorAll(
            '#kt_docs_repeater_plantravail input[name*="longueur"], #kt_docs_repeater_plantravail input[name*="largeur"], #kt_docs_repeater_plantravail select[name*="epaisseur"]',
        )
        .forEach(function (element) {
            element.addEventListener("change", updatePdtRecap);
            element.addEventListener("input", updatePdtRecap);
            element.addEventListener("change", calculateTotal);
            element.addEventListener("input", calculateTotal);
        });

    // Attacher les événements pour le récapitulatif des crédences aux éléments existants
    document
        .querySelectorAll(
            '#kt_docs_repeater_credence input[name*="credence_longueur"], #kt_docs_repeater_credence input[name*="credence_largeur"], #kt_docs_repeater_credence select[name*="credence_epaisseur"]',
        )
        .forEach(function (element) {
            element.addEventListener("change", updateCredenceRecap);
            element.addEventListener("input", updateCredenceRecap);
            element.addEventListener("change", calculateTotal);
            element.addEventListener("input", calculateTotal);
        });

    // Ajouter l'événement de changement pour addperc
    const addpercInput = document.getElementById("addperc");
    if (addpercInput) {
        addpercInput.addEventListener("input", function () {
            updateAllPricesWithAddperc();
        });
        addpercInput.addEventListener("change", function () {
            updateAllPricesWithAddperc();
        });
    }

    // Initialiser les récapitulatifs au chargement
    updatePdtRecap();
    updateCredenceRecap();
    updateOptionsRecap();

    // Observer les changements dans le container des plans de travail pour détecter les suppressions
    const pdtContainer = document.getElementById(
        "kt_docs_repeater_plantravail",
    );
    if (pdtContainer) {
        const observer = new MutationObserver(function (mutations) {
            let shouldUpdate = false;
            mutations.forEach(function (mutation) {
                if (mutation.type === "childList") {
                    // Détecter si des éléments ont été ajoutés ou supprimés
                    if (
                        mutation.removedNodes.length > 0 ||
                        mutation.addedNodes.length > 0
                    ) {
                        shouldUpdate = true;
                    }
                }
            });
            if (shouldUpdate) {
                setTimeout(updatePdtRecap, 100);
            }
        });

        observer.observe(pdtContainer, {
            childList: true,
            subtree: true,
        });
    }

    // Observer les changements dans le container des crédences pour détecter les suppressions
    const credenceContainer = document.getElementById(
        "kt_docs_repeater_credence",
    );
    if (credenceContainer) {
        const credenceObserver = new MutationObserver(function (mutations) {
            let shouldUpdate = false;
            mutations.forEach(function (mutation) {
                if (mutation.type === "childList") {
                    // Détecter si des élémentshave been added or supprimés
                    if (
                        mutation.removedNodes.length > 0 ||
                        mutation.addedNodes.length > 0
                    ) {
                        shouldUpdate = true;
                    }
                }
            });
            if (shouldUpdate) {
                setTimeout(updateCredenceRecap, 100);
            }
        });

        credenceObserver.observe(credenceContainer, {
            childList: true,
            subtree: true,
        });
    }

    // Attacher les événements pour le récapitulatif des options de découpe
    document
        .querySelectorAll(
            '.optionsdecoupe-quantity, select[name*="optionsdecoupe"]',
        )
        .forEach(function (select) {
            select.addEventListener("change", updateOptionsRecap);
            select.addEventListener("input", updateOptionsRecap);
        });

    // Initialiser le récapitulatif des services au chargement
    updateServicesRecap();

    // Attacher les événements pour le récapitulatif des services
    document
        .querySelectorAll('input[name*="services"], .service-checkbox')
        .forEach(function (checkbox) {
            checkbox.addEventListener("change", updateServicesRecap);
        });

    // Initialiser le récapitulatif de la TVA au chargement
    updateTvaRecap();

    // Attacher les événements pour le récapitulatif de la TVA
    document
        .querySelectorAll('input[name*="tva"], .tva-checkbox')
        .forEach(function (checkbox) {
            checkbox.addEventListener("change", function () {
                updateTvaRecap();
                calculateTotal(); // Recalculer le total TTC quand la TVA change
            });
        });

    // Attacher les événements de calcul à tous les éléments concernés
    attachCalculationEvents();

    // Calculer le total initial
    calculateTotal();

    // Appliquer addperc aux prix initiaux
    setTimeout(() => {
        updateAllPricesWithAddperc();
    }, 500);
});

// Fonction pour attacher les événements de calcul
function attachCalculationEvents() {
    // Plans de travail
    document
        .querySelectorAll(
            '#kt_docs_repeater_plantravail input[name*="longueur"], ' +
                '#kt_docs_repeater_plantravail input[name*="largeur"], ' +
                '#kt_docs_repeater_plantravail select[name*="epaisseur"]',
        )
        .forEach((element) => {
            element.addEventListener("change", calculateTotal);
            element.addEventListener("input", calculateTotal);
        });

    // Bordures
    const bordureSelect = document.getElementById("bordure_id");
    if (bordureSelect) {
        bordureSelect.addEventListener("change", calculateTotal);
    }

    // Crédences
    document
        .querySelectorAll(
            '#kt_docs_repeater_credence input[name*="credence_longueur"], ' +
                '#kt_docs_repeater_credence input[name*="largeur"], ' +
                '#kt_docs_repeater_credence select[name*="epaisseur"]',
        )
        .forEach((element) => {
            element.addEventListener("change", calculateTotal);
            element.addEventListener("input", calculateTotal);
        });

    // Options et découpes
    document
        .querySelectorAll(
            '.optionsdecoupe-quantity, select[name*="optionsdecoupe"]',
        )
        .forEach((select) => {
            select.addEventListener("change", calculateTotal);
        });

    // Services
    document
        .querySelectorAll('input[name*="services"], .service-checkbox')
        .forEach((checkbox) => {
            checkbox.addEventListener("change", calculateTotal);
        });
}

function calculateTotal() {
    console.log("🧮 === DÉBUT DU CALCUL DU TOTAL (CÔTÉ SERVEUR) ===");

    // Debug: vérifier si tous les éléments nécessaires sont présents
    console.log("📊 Éléments trouvés:");
    console.log(
        "- Plans de travail:",
        document.querySelectorAll(
            "#kt_docs_repeater_plantravail [data-repeater-item]",
        ).length,
    );
    console.log(
        "- Crédences:",
        document.querySelectorAll(
            "#kt_docs_repeater_credence [data-repeater-item]",
        ).length,
    );
    console.log(
        "- Options découpe:",
        document.querySelectorAll('input[name*="optionsdecoupes"]').length,
    );
    console.log(
        "- Services:",
        document.querySelectorAll('input[name*="services"]').length,
    );
    console.log(
        "- Bordure sélectionnée:",
        document.getElementById("bordure_id")?.value || "aucune",
    );

    // Collecter toutes les données du formulaire
    const formData = new FormData();

    // Plans de travail
    const plansTravail = [];
    document
        .querySelectorAll("#kt_docs_repeater_plantravail [data-repeater-item]")
        .forEach((item, index) => {
            const longueurInput = item.querySelector('input[name*="longueur"]');
            const largeurInput = item.querySelector('input[name*="largeur"]');
            const epaisseurSelect = item.querySelector(
                'select[name*="epaisseur"]',
            );

            if (longueurInput && largeurInput && epaisseurSelect) {
                const longueur = parseFloat(longueurInput.value) || 0;
                const largeur = parseFloat(largeurInput.value) || 0;
                const epaisseur = epaisseurSelect.value;

                if (longueur > 0 && largeur > 0 && epaisseur) {
                    plansTravail.push({
                        "pl-longueur": longueur,
                        "pl-largeur": largeur,
                        "pl-epaisseur": epaisseur,
                    });
                }
            }
        });

    // Crédences
    const credences = [];
    document
        .querySelectorAll("#kt_docs_repeater_credence [data-repeater-item]")
        .forEach((item, index) => {
            const longueurInput = item.querySelector(
                'input[name*="credence_longueur"]',
            );
            const largeurInput = item.querySelector(
                'input[name*="credence_largeur"]',
            );
            const epaisseurSelect = item.querySelector(
                'select[name*="credence_epaisseur"]',
            );

            if (longueurInput && largeurInput && epaisseurSelect) {
                const longueur = parseFloat(longueurInput.value) || 0;
                const largeur = parseFloat(largeurInput.value) || 0;
                const epaisseur = epaisseurSelect.value;

                if (longueur > 0 && largeur > 0 && epaisseur) {
                    credences.push({
                        credence_longueur: longueur,
                        credence_largeur: largeur,
                        credence_epaisseur: epaisseur,
                    });
                }
            }
        });

    // Options de découpe
    const optionsDecoupes = {};
    document
        .querySelectorAll(
            'input[name*="optionsdecoupes"], .optionsdecoupe-quantity',
        )
        .forEach((input) => {
            const name = input.name;
            if (name) {
                const match = name.match(/optionsdecoupes\[(\d+)\]/);
                if (match && input.value && parseFloat(input.value) > 0) {
                    optionsDecoupes[match[1]] = parseFloat(input.value);
                }
            }
        });

    // Services
    const services = {};
    document
        .querySelectorAll('input[name*="services"], .service-checkbox')
        .forEach((checkbox) => {
            const name = checkbox.name;
            if (name) {
                const match = name.match(/services\[(\d+)\]/);
                if (match) {
                    services[match[1]] = checkbox.checked ? 1 : 0;
                }
            }
        });

    // TVA
    const tvaCheckbox = document.querySelector('input[name="tva"]');
    let tva = 20;
    if (tvaCheckbox) {
        tva = tvaCheckbox.checked ? 10 : 20;
    } else {
        // Fallback: lire depuis l'élément d'affichage
        const tvaElement = document.getElementById("tva-id");
        if (tvaElement && tvaElement.textContent.includes("10%")) {
            tva = 10;
        }
    }

    // Bordure
    const bordureSelect = document.getElementById("bordure_id");
    let bordureId = null;
    if (bordureSelect && bordureSelect.value) {
        bordureId = bordureSelect.value;
    }

    // ID du devis pour récupérer le montant original (en mode édition)
    const devisId = document.querySelector('input[name="_method"]')
        ? window.location.pathname.split("/").pop()
        : null;

    // Récupérer la valeur addperc
    const addpercInput = document.getElementById("addperc");
    let addperc = 0;
    if (addpercInput && addpercInput.value) {
        addperc = parseInt(addpercInput.value) || 0;
    }

    // Envoyer la requête AJAX
    fetch("/admin/devis/calculate-total", {
        method: "POST",
        headers: {
            "Content-Type": "application/json",
            "X-CSRF-TOKEN": document
                .querySelector('meta[name="csrf-token"]')
                .getAttribute("content"),
        },
        body: JSON.stringify({
            kt_docs_repeater_plantravail: plansTravail,
            kt_docs_repeater_credence: credences,
            optionsdecoupes: optionsDecoupes,
            services: services,
            bordure_id: bordureId,
            addperc: addperc,
            tva: tva,
            devis_id: devisId,
        }),
    })
        .then((response) => response.json())
        .then((data) => {
            if (data.success) {
                console.log(
                    `💰 Total calculé côté serveur: ${data.total_ht}€ HT`,
                );

                // Mettre à jour l'affichage
                const totalAmountElement =
                    document.getElementById("total-amount");
                if (totalAmountElement) {
                    totalAmountElement.textContent = data.total_ht_formatted;
                }

                const totalTTCElement = document.getElementById("total-ttc");
                if (totalTTCElement) {
                    totalTTCElement.textContent = data.total_ttc_formatted;
                }

                // Mettre à jour l'input hidden
                let totalHtInput = document.getElementById(
                    "total_ht_calculated",
                );
                if (!totalHtInput) {
                    // Créer l'input s'il n'existe pas
                    totalHtInput = document.createElement("input");
                    totalHtInput.type = "hidden";
                    totalHtInput.id = "total_ht_calculated";
                    totalHtInput.name = "total_ht_calculated";
                    document.querySelector("form").appendChild(totalHtInput);
                }
                totalHtInput.value = data.total_ht;

                // Vérifier les différences
                checkMontantDifference(
                    parseFloat(data.total_ht),
                    data.montant_original,
                );
            } else {
                console.error("Erreur lors du calcul:", data.message);
            }
        })
        .catch((error) => {
            console.error("Erreur de réseau:", error);
        });

    console.log("🧮 === FIN DU CALCUL DU TOTAL ===");
}

// Fonction pour vérifier la différence entre le montant calculé et enregistré
function checkMontantDifference(calculatedTotal, originalAmount = null) {
    const isEditMode =
        $('input[name="_method"]').length > 0 &&
        $('input[name="_method"]').val() === "PUT";

    if (isEditMode) {
        let savedMontant = originalAmount;

        // Si pas fourni par le serveur, essayer de récupérer depuis l'input hidden
        if (savedMontant === null || savedMontant === 0) {
            const savedMontantInput = document.querySelector(
                'input[name="montant_ht_original"]',
            );
            if (savedMontantInput && savedMontantInput.value) {
                savedMontant = parseFloat(savedMontantInput.value) || 0;
            }
        }

        console.log(
            `🔍 Comparaison: Calculé=${calculatedTotal.toFixed(2)}€, Original=${savedMontant ? savedMontant.toFixed(2) : "N/A"}€`,
        );

        if (savedMontant && savedMontant > 0) {
            // Comparer avec une tolérance de 0.01€ pour éviter les erreurs d'arrondi
            if (Math.abs(calculatedTotal - savedMontant) > 0.01) {
                console.log(
                    `⚠️ Différence détectée: Calculé=${calculatedTotal.toFixed(2)}€, Enregistré=${savedMontant.toFixed(2)}€`,
                );
                //KHA  showOldTotalRecap(savedMontant, calculatedTotal);
            } else {
                hideOldTotalRecap();
            }
        }
    } else {
        hideOldTotalRecap();
    }
}

// Fonction pour afficher le récapitulatif de l'ancien total
function showOldTotalRecap(oldTotal, newTotal) {
    let oldTotalContainer = document.getElementById("old-total-recap");

    // Créer le container s'il n'existe pas
    if (!oldTotalContainer) {
        oldTotalContainer = document.createElement("div");
        oldTotalContainer.id = "old-total-recap";
        oldTotalContainer.className = "alert alert-warning mt-3";

        // Insérer avant le total HT actuel
        const totalContainer = document
            .getElementById("total-amount")
            ?.closest(".card-body");
        if (totalContainer) {
            totalContainer.insertBefore(
                oldTotalContainer,
                totalContainer.firstChild,
            );
        }
    }

    const difference = newTotal - oldTotal;
    const differenceText =
        difference > 0
            ? `+${difference.toFixed(2)}€`
            : `${difference.toFixed(2)}€`;
    const differenceClass = difference > 0 ? "text-success" : "text-danger";

    oldTotalContainer.innerHTML = `
        <div class="d-flex align-items-center">
            <i class="ki-duotone ki-information-5 fs-2 text-warning me-3">
                <span class="path1"></span>
                <span class="path2"></span>
                <span class="path3"></span>
            </i>
            <div class="flex-grow-1">
                <div class="fw-bold">Modification du montant détectée</div>
                <div class="fs-7 text-muted">
                    Ancien total HT : <strong>${oldTotal.toFixed(2)}€</strong><br>
                    Nouveau total HT : <strong>${newTotal.toFixed(2)}€</strong><br>
                    Différence : <span class="${differenceClass} fw-bold">${differenceText}</span>
                </div>
            </div>
        </div>
    `;

    oldTotalContainer.style.display = "block";
}

// Fonction pour masquer le récapitulatif de l'ancien total
function hideOldTotalRecap() {
    const oldTotalContainer = document.getElementById("old-total-recap");
    if (oldTotalContainer) {
        oldTotalContainer.style.display = "none";
    }
}

// Fonction pour mettre à jour le récapitulatif des options de découpe
function updateOptionsRecap() {
    const optionsContainer = document.getElementById("options-id");
    if (!optionsContainer) return;

    // Vider le conteneur
    optionsContainer.innerHTML = "";

    let optionIndex = 1;
    let hasValidOptions = false;
    const addperc = getAddpercValue();

    // Récupérer tous les selects de quantité des options de découpe
    const quantitySelects = document.querySelectorAll(
        '.optionsdecoupe-quantity, select[name*="optionsdecoupe"]',
    );

    quantitySelects.forEach((select) => {
        const quantity = parseInt(select.value) || 0;

        // Vérifier si la quantité est valide (supérieure à 0)
        if (quantity > 0) {
            const prixOriginal =
                parseFloat(select.getAttribute("data-prix")) || 0;
            const prixAvecPourcentage = applyAddpercToPrice(
                prixOriginal,
                addperc,
            );
            const optionText =
                select
                    .closest(".col-md-3, .form-group, .mb-3")
                    ?.querySelector(".form-label, label")
                    ?.textContent?.trim() ||
                select
                    .closest(".row")
                    ?.querySelector(".form-label, label")
                    ?.textContent?.trim() ||
                "";

            if (optionText) {
                const totalPrix = quantity * prixAvecPourcentage;

                // Créer l'élément récapitulatif
                const recapDiv = document.createElement("div");
                recapDiv.className = "option-name mb-2";
                recapDiv.innerHTML = `
                    <div class="fw-bold mt-5">Options et découpes ${optionIndex}</div>
                    <div class="text-gray-600">${optionText} (Qté: ${quantity})</div>
                    <div class="text-success fw-bold">${totalPrix.toFixed(2)}€</div>
                `;

                optionsContainer.appendChild(recapDiv);
                optionIndex++;
                hasValidOptions = true;
            }
        }
    });

    // Si aucune option valide, afficher un message par défaut
    if (!hasValidOptions) {
        optionsContainer.innerHTML =
            '<div class="option-name text-muted"></div>';
    }
}

// Fonction pour mettre à jour le récapitulatif des services
function updateServicesRecap() {
    const servicesContainer = document.getElementById("services-id");
    if (!servicesContainer) return;

    // Vider le conteneur
    servicesContainer.innerHTML = "";

    let serviceIndex = 1;
    let hasValidServices = false;
    const addperc = getAddpercValue();

    // Récupérer tous les checkboxes de services cochés
    const checkedServices = document.querySelectorAll(
        'input[name*="services"]:checked, .service-checkbox:checked',
    );

    checkedServices.forEach((checkbox) => {
        const prixOriginal =
            parseFloat(checkbox.getAttribute("data-prix")) || 0;
        const prixAvecPourcentage = applyAddpercToPrice(prixOriginal, addperc);
        const serviceText =
            checkbox
                .closest(".form-check")
                ?.querySelector("label")
                ?.textContent?.trim() ||
            checkbox
                .closest(".col-md-6")
                ?.querySelector("label")
                ?.textContent?.trim() ||
            "";

        if (serviceText) {
            // Créer l'élément récapitulatif
            const recapDiv = document.createElement("div");
            recapDiv.className = "service-name mb-2";
            recapDiv.innerHTML = `
                <div class="fw-bold mt-5">Service ${serviceIndex}</div>
                <div class="text-gray-600">${serviceText}</div>
                <div class="text-success fw-bold">${prixAvecPourcentage.toFixed(2)}€</div>
            `;

            servicesContainer.appendChild(recapDiv);
            serviceIndex++;
            hasValidServices = true;
        }
    });

    // Si aucun service sélectionné, afficher un message par défaut
    if (!hasValidServices) {
        servicesContainer.innerHTML =
            '<div class="service-name text-muted"></div>';
    }
}

// Fonction pour mettre à jour le récapitulatif de la TVA
function updateTvaRecap() {
    const tvaContainer = document.getElementById("tva-id");
    if (!tvaContainer) return;

    // Récupérer la checkbox de TVA
    const tvaCheckbox = document.querySelector('input[name="tva"]');

    if (tvaCheckbox) {
        // Vérifier si la checkbox est cochée
        const isChecked = tvaCheckbox.checked;
        const tvaRate = isChecked ? "10%" : "20%";

        // Mettre à jour le contenu avec le taux approprié
        tvaContainer.textContent = tvaRate;
    } else {
        // Si pas de checkbox trouvée, afficher 20% par défaut
        tvaContainer.textContent = "20%";
    }
}

// Fonction pour obtenir la valeur du pourcentage addperc
function getAddpercValue() {
    const addpercInput = document.getElementById("addperc");
    return addpercInput ? parseInt(addpercInput.value) || 0 : 0;
}

// Fonction pour appliquer le pourcentage addperc à un prix
function applyAddpercToPrice(originalPrice, addperc) {
    return originalPrice * (1 + addperc / 100);
}

// Fonction pour mettre à jour tous les prix avec addperc
function updateAllPricesWithAddperc() {
    const addperc = getAddpercValue();

    // Mettre à jour les prix des épaisseurs dans tous les selects
    updateEpaisseurPrices(addperc);

    // Mettre à jour les prix des bordures
    updateBordurePrices(addperc);

    // Mettre à jour les prix des options de découpe
    updateOptionsPrices(addperc);

    // Mettre à jour les prix des services
    updateServicesPrices(addperc);

    // Recalculer tous les récapitulatifs
    updatePdtRecap();
    updateCredenceRecap();
    updateOptionsRecap();
    updateServicesRecap();
    updateBordureRecap();

    // Recalculer le total
    calculateTotal();
}

// Fonction pour mettre à jour les prix des épaisseurs
function updateEpaisseurPrices(addperc) {
    const epaisseurSelects = document.querySelectorAll("select.epaisseur");

    epaisseurSelects.forEach((select) => {
        const selectedValue = select.value; // Sauvegarder la valeur sélectionnée

        Array.from(select.options).forEach((option) => {
            if (option.value && option.value !== "") {
                const originalPrice =
                    parseFloat(option.getAttribute("data-prix")) || 0;
                if (originalPrice > 0) {
                    const newPrice = applyAddpercToPrice(
                        originalPrice,
                        addperc,
                    );

                    // Mettre à jour le texte de l'option
                    const optionText = option.textContent;
                    const baseText = optionText.split(" - ")[0]; // Garder seulement la partie avant le prix
                    option.textContent = `${baseText} - ${newPrice.toFixed(2)}€/m²`;
                }
            }
        });

        // Réinitialiser Select2 pour afficher les nouveaux prix
        if ($(select).hasClass("select2-hidden-accessible")) {
            $(select).select2("destroy");
            $(select).select2({
                minimumResultsForSearch: Infinity,
            });
            // Remettre la valeur sélectionnée
            $(select).val(selectedValue).trigger("change");
        }
    });
}

// Fonction pour mettre à jour les prix des bordures
function updateBordurePrices(addperc) {
    const bordureSelect = document.getElementById("bordure_id");
    if (!bordureSelect) return;

    const selectedValue = bordureSelect.value; // Sauvegarder la valeur sélectionnée

    Array.from(bordureSelect.options).forEach((option) => {
        if (option.value && option.value !== "") {
            const originalPrice =
                parseFloat(option.getAttribute("data-prix")) || 0;
            if (originalPrice > 0) {
                const newPrice = applyAddpercToPrice(originalPrice, addperc);

                // Mettre à jour le texte de l'option
                const optionText = option.textContent;
                const parts = optionText.split(" - ");
                if (parts.length > 1) {
                    // Si le prix est déjà dans le texte, le remplacer
                    option.textContent = `${parts[0]} - ${newPrice.toFixed(2)}€`;
                }
            }
        }
    });

    // Réinitialiser Select2 pour afficher les nouveaux prix
    if ($(bordureSelect).hasClass("select2-hidden-accessible")) {
        $(bordureSelect).select2("destroy");
        $(bordureSelect).select2({
            minimumResultsForSearch: Infinity,
        });
        // Remettre la valeur sélectionnée
        $(bordureSelect).val(selectedValue).trigger("change");
    }
}

// Fonction pour mettre à jour les prix des options de découpe
function updateOptionsPrices(addperc) {
    const optionElements = document.querySelectorAll(
        ".optionsdecoupe-quantity",
    );

    optionElements.forEach((select) => {
        const originalPrice = parseFloat(select.getAttribute("data-prix")) || 0;
        if (originalPrice > 0) {
            const newPrice = applyAddpercToPrice(originalPrice, addperc);

            // Trouver l'élément d'affichage du prix (div avec class text-muted)
            const priceElement = select
                .closest(".col-md-3")
                ?.querySelector(".text-muted");
            if (priceElement) {
                priceElement.textContent = `${newPrice.toFixed(2)}€`;
            }
        }
    });
}

// Fonction pour mettre à jour les prix des services
function updateServicesPrices(addperc) {
    const serviceCheckboxes = document.querySelectorAll(".service-checkbox");

    serviceCheckboxes.forEach((checkbox) => {
        const originalPrice =
            parseFloat(checkbox.getAttribute("data-prix")) || 0;
        if (originalPrice > 0) {
            const newPrice = applyAddpercToPrice(originalPrice, addperc);

            // Trouver l'élément d'affichage du prix
            const priceElement = checkbox
                .closest("label")
                ?.querySelector(".text-success");
            if (priceElement) {
                priceElement.textContent = `: ${newPrice.toFixed(2)}€`;
            }
        }
    });
}
